// Copyright (C) 2016 Free Software Foundation, Inc.
//
// This file is part of the GNU ISO C++ Library.  This library is free
// software; you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the
// Free Software Foundation; either version 3, or (at your option)
// any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License along
// with this library; see the file COPYING3.  If not see
// <http://www.gnu.org/licenses/>.

//  hermite
//  Compare against values generated by the GNU Scientific Library.
//  The GSL can be found on the web: http://www.gnu.org/software/gsl/
#include <limits>
#include <tr1/cmath>
#if defined(__TEST_DEBUG)
#  include <iostream>
#  define VERIFY(A) \
  if (!(A)) \
    { \
      std::cout << "line " << __LINE__ \
	<< "  max_abs_frac = " << max_abs_frac \
	<< std::endl; \
    }
#else
#  include <testsuite_hooks.h>
#endif
#include <specfun_testcase.h>


// Test data for n=0.
// max(|f - f_GSL|): 0.0000000000000000
// max(|f - f_GSL| / |f_GSL|): 0.0000000000000000
testcase_hermite<double>
data001[] =
{
  { 1.0000000000000000, 0, -10.000000000000000 },
  { 1.0000000000000000, 0, -9.9000000000000004 },
  { 1.0000000000000000, 0, -9.8000000000000007 },
  { 1.0000000000000000, 0, -9.6999999999999993 },
  { 1.0000000000000000, 0, -9.5999999999999996 },
  { 1.0000000000000000, 0, -9.5000000000000000 },
  { 1.0000000000000000, 0, -9.4000000000000004 },
  { 1.0000000000000000, 0, -9.3000000000000007 },
  { 1.0000000000000000, 0, -9.1999999999999993 },
  { 1.0000000000000000, 0, -9.0999999999999996 },
  { 1.0000000000000000, 0, -9.0000000000000000 },
  { 1.0000000000000000, 0, -8.9000000000000004 },
  { 1.0000000000000000, 0, -8.8000000000000007 },
  { 1.0000000000000000, 0, -8.6999999999999993 },
  { 1.0000000000000000, 0, -8.5999999999999996 },
  { 1.0000000000000000, 0, -8.5000000000000000 },
  { 1.0000000000000000, 0, -8.4000000000000004 },
  { 1.0000000000000000, 0, -8.3000000000000007 },
  { 1.0000000000000000, 0, -8.1999999999999993 },
  { 1.0000000000000000, 0, -8.0999999999999996 },
  { 1.0000000000000000, 0, -8.0000000000000000 },
  { 1.0000000000000000, 0, -7.9000000000000004 },
  { 1.0000000000000000, 0, -7.7999999999999998 },
  { 1.0000000000000000, 0, -7.7000000000000002 },
  { 1.0000000000000000, 0, -7.5999999999999996 },
  { 1.0000000000000000, 0, -7.5000000000000000 },
  { 1.0000000000000000, 0, -7.4000000000000004 },
  { 1.0000000000000000, 0, -7.2999999999999998 },
  { 1.0000000000000000, 0, -7.2000000000000002 },
  { 1.0000000000000000, 0, -7.0999999999999996 },
  { 1.0000000000000000, 0, -7.0000000000000000 },
  { 1.0000000000000000, 0, -6.9000000000000004 },
  { 1.0000000000000000, 0, -6.7999999999999998 },
  { 1.0000000000000000, 0, -6.7000000000000002 },
  { 1.0000000000000000, 0, -6.5999999999999996 },
  { 1.0000000000000000, 0, -6.5000000000000000 },
  { 1.0000000000000000, 0, -6.4000000000000004 },
  { 1.0000000000000000, 0, -6.2999999999999998 },
  { 1.0000000000000000, 0, -6.2000000000000002 },
  { 1.0000000000000000, 0, -6.0999999999999996 },
  { 1.0000000000000000, 0, -6.0000000000000000 },
  { 1.0000000000000000, 0, -5.9000000000000004 },
  { 1.0000000000000000, 0, -5.7999999999999998 },
  { 1.0000000000000000, 0, -5.7000000000000002 },
  { 1.0000000000000000, 0, -5.5999999999999996 },
  { 1.0000000000000000, 0, -5.5000000000000000 },
  { 1.0000000000000000, 0, -5.4000000000000004 },
  { 1.0000000000000000, 0, -5.2999999999999998 },
  { 1.0000000000000000, 0, -5.2000000000000002 },
  { 1.0000000000000000, 0, -5.0999999999999996 },
  { 1.0000000000000000, 0, -5.0000000000000000 },
  { 1.0000000000000000, 0, -4.9000000000000004 },
  { 1.0000000000000000, 0, -4.7999999999999998 },
  { 1.0000000000000000, 0, -4.7000000000000002 },
  { 1.0000000000000000, 0, -4.5999999999999996 },
  { 1.0000000000000000, 0, -4.5000000000000000 },
  { 1.0000000000000000, 0, -4.4000000000000004 },
  { 1.0000000000000000, 0, -4.2999999999999998 },
  { 1.0000000000000000, 0, -4.2000000000000002 },
  { 1.0000000000000000, 0, -4.0999999999999996 },
  { 1.0000000000000000, 0, -4.0000000000000000 },
  { 1.0000000000000000, 0, -3.9000000000000004 },
  { 1.0000000000000000, 0, -3.7999999999999998 },
  { 1.0000000000000000, 0, -3.7000000000000002 },
  { 1.0000000000000000, 0, -3.5999999999999996 },
  { 1.0000000000000000, 0, -3.5000000000000000 },
  { 1.0000000000000000, 0, -3.4000000000000004 },
  { 1.0000000000000000, 0, -3.2999999999999998 },
  { 1.0000000000000000, 0, -3.2000000000000002 },
  { 1.0000000000000000, 0, -3.0999999999999996 },
  { 1.0000000000000000, 0, -3.0000000000000000 },
  { 1.0000000000000000, 0, -2.9000000000000004 },
  { 1.0000000000000000, 0, -2.7999999999999998 },
  { 1.0000000000000000, 0, -2.7000000000000002 },
  { 1.0000000000000000, 0, -2.5999999999999996 },
  { 1.0000000000000000, 0, -2.5000000000000000 },
  { 1.0000000000000000, 0, -2.4000000000000004 },
  { 1.0000000000000000, 0, -2.2999999999999998 },
  { 1.0000000000000000, 0, -2.2000000000000002 },
  { 1.0000000000000000, 0, -2.0999999999999996 },
  { 1.0000000000000000, 0, -2.0000000000000000 },
  { 1.0000000000000000, 0, -1.9000000000000004 },
  { 1.0000000000000000, 0, -1.8000000000000007 },
  { 1.0000000000000000, 0, -1.6999999999999993 },
  { 1.0000000000000000, 0, -1.5999999999999996 },
  { 1.0000000000000000, 0, -1.5000000000000000 },
  { 1.0000000000000000, 0, -1.4000000000000004 },
  { 1.0000000000000000, 0, -1.3000000000000007 },
  { 1.0000000000000000, 0, -1.1999999999999993 },
  { 1.0000000000000000, 0, -1.0999999999999996 },
  { 1.0000000000000000, 0, -1.0000000000000000 },
  { 1.0000000000000000, 0, -0.90000000000000036 },
  { 1.0000000000000000, 0, -0.80000000000000071 },
  { 1.0000000000000000, 0, -0.69999999999999929 },
  { 1.0000000000000000, 0, -0.59999999999999964 },
  { 1.0000000000000000, 0, -0.50000000000000000 },
  { 1.0000000000000000, 0, -0.40000000000000036 },
  { 1.0000000000000000, 0, -0.30000000000000071 },
  { 1.0000000000000000, 0, -0.19999999999999929 },
  { 1.0000000000000000, 0, -0.099999999999999645 },
  { 1.0000000000000000, 0, 0.0000000000000000 },
  { 1.0000000000000000, 0, 0.099999999999999645 },
  { 1.0000000000000000, 0, 0.19999999999999929 },
  { 1.0000000000000000, 0, 0.30000000000000071 },
  { 1.0000000000000000, 0, 0.40000000000000036 },
  { 1.0000000000000000, 0, 0.50000000000000000 },
  { 1.0000000000000000, 0, 0.59999999999999964 },
  { 1.0000000000000000, 0, 0.69999999999999929 },
  { 1.0000000000000000, 0, 0.80000000000000071 },
  { 1.0000000000000000, 0, 0.90000000000000036 },
  { 1.0000000000000000, 0, 1.0000000000000000 },
  { 1.0000000000000000, 0, 1.0999999999999996 },
  { 1.0000000000000000, 0, 1.1999999999999993 },
  { 1.0000000000000000, 0, 1.3000000000000007 },
  { 1.0000000000000000, 0, 1.4000000000000004 },
  { 1.0000000000000000, 0, 1.5000000000000000 },
  { 1.0000000000000000, 0, 1.5999999999999996 },
  { 1.0000000000000000, 0, 1.6999999999999993 },
  { 1.0000000000000000, 0, 1.8000000000000007 },
  { 1.0000000000000000, 0, 1.9000000000000004 },
  { 1.0000000000000000, 0, 2.0000000000000000 },
  { 1.0000000000000000, 0, 2.0999999999999996 },
  { 1.0000000000000000, 0, 2.1999999999999993 },
  { 1.0000000000000000, 0, 2.3000000000000007 },
  { 1.0000000000000000, 0, 2.4000000000000004 },
  { 1.0000000000000000, 0, 2.5000000000000000 },
  { 1.0000000000000000, 0, 2.5999999999999996 },
  { 1.0000000000000000, 0, 2.6999999999999993 },
  { 1.0000000000000000, 0, 2.8000000000000007 },
  { 1.0000000000000000, 0, 2.9000000000000004 },
  { 1.0000000000000000, 0, 3.0000000000000000 },
  { 1.0000000000000000, 0, 3.0999999999999996 },
  { 1.0000000000000000, 0, 3.1999999999999993 },
  { 1.0000000000000000, 0, 3.3000000000000007 },
  { 1.0000000000000000, 0, 3.4000000000000004 },
  { 1.0000000000000000, 0, 3.5000000000000000 },
  { 1.0000000000000000, 0, 3.5999999999999996 },
  { 1.0000000000000000, 0, 3.6999999999999993 },
  { 1.0000000000000000, 0, 3.8000000000000007 },
  { 1.0000000000000000, 0, 3.9000000000000004 },
  { 1.0000000000000000, 0, 4.0000000000000000 },
  { 1.0000000000000000, 0, 4.0999999999999996 },
  { 1.0000000000000000, 0, 4.1999999999999993 },
  { 1.0000000000000000, 0, 4.3000000000000007 },
  { 1.0000000000000000, 0, 4.4000000000000004 },
  { 1.0000000000000000, 0, 4.5000000000000000 },
  { 1.0000000000000000, 0, 4.5999999999999996 },
  { 1.0000000000000000, 0, 4.6999999999999993 },
  { 1.0000000000000000, 0, 4.8000000000000007 },
  { 1.0000000000000000, 0, 4.9000000000000004 },
  { 1.0000000000000000, 0, 5.0000000000000000 },
  { 1.0000000000000000, 0, 5.0999999999999996 },
  { 1.0000000000000000, 0, 5.1999999999999993 },
  { 1.0000000000000000, 0, 5.3000000000000007 },
  { 1.0000000000000000, 0, 5.4000000000000004 },
  { 1.0000000000000000, 0, 5.5000000000000000 },
  { 1.0000000000000000, 0, 5.5999999999999996 },
  { 1.0000000000000000, 0, 5.6999999999999993 },
  { 1.0000000000000000, 0, 5.8000000000000007 },
  { 1.0000000000000000, 0, 5.9000000000000004 },
  { 1.0000000000000000, 0, 6.0000000000000000 },
  { 1.0000000000000000, 0, 6.1000000000000014 },
  { 1.0000000000000000, 0, 6.1999999999999993 },
  { 1.0000000000000000, 0, 6.3000000000000007 },
  { 1.0000000000000000, 0, 6.3999999999999986 },
  { 1.0000000000000000, 0, 6.5000000000000000 },
  { 1.0000000000000000, 0, 6.6000000000000014 },
  { 1.0000000000000000, 0, 6.6999999999999993 },
  { 1.0000000000000000, 0, 6.8000000000000007 },
  { 1.0000000000000000, 0, 6.8999999999999986 },
  { 1.0000000000000000, 0, 7.0000000000000000 },
  { 1.0000000000000000, 0, 7.1000000000000014 },
  { 1.0000000000000000, 0, 7.1999999999999993 },
  { 1.0000000000000000, 0, 7.3000000000000007 },
  { 1.0000000000000000, 0, 7.3999999999999986 },
  { 1.0000000000000000, 0, 7.5000000000000000 },
  { 1.0000000000000000, 0, 7.6000000000000014 },
  { 1.0000000000000000, 0, 7.6999999999999993 },
  { 1.0000000000000000, 0, 7.8000000000000007 },
  { 1.0000000000000000, 0, 7.8999999999999986 },
  { 1.0000000000000000, 0, 8.0000000000000000 },
  { 1.0000000000000000, 0, 8.1000000000000014 },
  { 1.0000000000000000, 0, 8.1999999999999993 },
  { 1.0000000000000000, 0, 8.3000000000000007 },
  { 1.0000000000000000, 0, 8.3999999999999986 },
  { 1.0000000000000000, 0, 8.5000000000000000 },
  { 1.0000000000000000, 0, 8.6000000000000014 },
  { 1.0000000000000000, 0, 8.6999999999999993 },
  { 1.0000000000000000, 0, 8.8000000000000007 },
  { 1.0000000000000000, 0, 8.8999999999999986 },
  { 1.0000000000000000, 0, 9.0000000000000000 },
  { 1.0000000000000000, 0, 9.1000000000000014 },
  { 1.0000000000000000, 0, 9.1999999999999993 },
  { 1.0000000000000000, 0, 9.3000000000000007 },
  { 1.0000000000000000, 0, 9.3999999999999986 },
  { 1.0000000000000000, 0, 9.5000000000000000 },
  { 1.0000000000000000, 0, 9.6000000000000014 },
  { 1.0000000000000000, 0, 9.6999999999999993 },
  { 1.0000000000000000, 0, 9.8000000000000007 },
  { 1.0000000000000000, 0, 9.8999999999999986 },
  { 1.0000000000000000, 0, 10.000000000000000 },
};

// Test function for n=0.
template<typename Tp>
  void
  test001()
  {
    bool test __attribute__((unused)) = true;
    const Tp eps = std::numeric_limits<Tp>::epsilon();
    Tp max_abs_diff = -Tp(1);
    Tp max_abs_frac = -Tp(1);
    unsigned int num_datum = sizeof(data001)
			   / sizeof(testcase_hermite<double>);
    for (unsigned int i = 0; i < num_datum; ++i)
      {
	const Tp f = std::tr1::hermite(Tp(data001[i].n), Tp(data001[i].x));
	const Tp f0 = data001[i].f0;
	const Tp diff = f - f0;
	if (std::abs(diff) > max_abs_diff)
	  max_abs_diff = std::abs(diff);
	if (std::abs(f0) > Tp(10) * eps
	 && std::abs(f) > Tp(10) * eps)
	  {
	    const Tp frac = diff / f0;
	    if (std::abs(frac) > max_abs_frac)
	      max_abs_frac = std::abs(frac);
	  }
      }
    VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
  }

// Test data for n=1.
// max(|f - f_GSL|): 0.0000000000000000
// max(|f - f_GSL| / |f_GSL|): 0.0000000000000000
testcase_hermite<double>
data002[] =
{
  { -20.000000000000000, 1, -10.000000000000000 },
  { -19.800000000000001, 1, -9.9000000000000004 },
  { -19.600000000000001, 1, -9.8000000000000007 },
  { -19.399999999999999, 1, -9.6999999999999993 },
  { -19.199999999999999, 1, -9.5999999999999996 },
  { -19.000000000000000, 1, -9.5000000000000000 },
  { -18.800000000000001, 1, -9.4000000000000004 },
  { -18.600000000000001, 1, -9.3000000000000007 },
  { -18.399999999999999, 1, -9.1999999999999993 },
  { -18.199999999999999, 1, -9.0999999999999996 },
  { -18.000000000000000, 1, -9.0000000000000000 },
  { -17.800000000000001, 1, -8.9000000000000004 },
  { -17.600000000000001, 1, -8.8000000000000007 },
  { -17.399999999999999, 1, -8.6999999999999993 },
  { -17.199999999999999, 1, -8.5999999999999996 },
  { -17.000000000000000, 1, -8.5000000000000000 },
  { -16.800000000000001, 1, -8.4000000000000004 },
  { -16.600000000000001, 1, -8.3000000000000007 },
  { -16.399999999999999, 1, -8.1999999999999993 },
  { -16.199999999999999, 1, -8.0999999999999996 },
  { -16.000000000000000, 1, -8.0000000000000000 },
  { -15.800000000000001, 1, -7.9000000000000004 },
  { -15.600000000000000, 1, -7.7999999999999998 },
  { -15.400000000000000, 1, -7.7000000000000002 },
  { -15.199999999999999, 1, -7.5999999999999996 },
  { -15.000000000000000, 1, -7.5000000000000000 },
  { -14.800000000000001, 1, -7.4000000000000004 },
  { -14.600000000000000, 1, -7.2999999999999998 },
  { -14.400000000000000, 1, -7.2000000000000002 },
  { -14.199999999999999, 1, -7.0999999999999996 },
  { -14.000000000000000, 1, -7.0000000000000000 },
  { -13.800000000000001, 1, -6.9000000000000004 },
  { -13.600000000000000, 1, -6.7999999999999998 },
  { -13.400000000000000, 1, -6.7000000000000002 },
  { -13.199999999999999, 1, -6.5999999999999996 },
  { -13.000000000000000, 1, -6.5000000000000000 },
  { -12.800000000000001, 1, -6.4000000000000004 },
  { -12.600000000000000, 1, -6.2999999999999998 },
  { -12.400000000000000, 1, -6.2000000000000002 },
  { -12.199999999999999, 1, -6.0999999999999996 },
  { -12.000000000000000, 1, -6.0000000000000000 },
  { -11.800000000000001, 1, -5.9000000000000004 },
  { -11.600000000000000, 1, -5.7999999999999998 },
  { -11.400000000000000, 1, -5.7000000000000002 },
  { -11.199999999999999, 1, -5.5999999999999996 },
  { -11.000000000000000, 1, -5.5000000000000000 },
  { -10.800000000000001, 1, -5.4000000000000004 },
  { -10.600000000000000, 1, -5.2999999999999998 },
  { -10.400000000000000, 1, -5.2000000000000002 },
  { -10.199999999999999, 1, -5.0999999999999996 },
  { -10.000000000000000, 1, -5.0000000000000000 },
  { -9.8000000000000007, 1, -4.9000000000000004 },
  { -9.5999999999999996, 1, -4.7999999999999998 },
  { -9.4000000000000004, 1, -4.7000000000000002 },
  { -9.1999999999999993, 1, -4.5999999999999996 },
  { -9.0000000000000000, 1, -4.5000000000000000 },
  { -8.8000000000000007, 1, -4.4000000000000004 },
  { -8.5999999999999996, 1, -4.2999999999999998 },
  { -8.4000000000000004, 1, -4.2000000000000002 },
  { -8.1999999999999993, 1, -4.0999999999999996 },
  { -8.0000000000000000, 1, -4.0000000000000000 },
  { -7.8000000000000007, 1, -3.9000000000000004 },
  { -7.5999999999999996, 1, -3.7999999999999998 },
  { -7.4000000000000004, 1, -3.7000000000000002 },
  { -7.1999999999999993, 1, -3.5999999999999996 },
  { -7.0000000000000000, 1, -3.5000000000000000 },
  { -6.8000000000000007, 1, -3.4000000000000004 },
  { -6.5999999999999996, 1, -3.2999999999999998 },
  { -6.4000000000000004, 1, -3.2000000000000002 },
  { -6.1999999999999993, 1, -3.0999999999999996 },
  { -6.0000000000000000, 1, -3.0000000000000000 },
  { -5.8000000000000007, 1, -2.9000000000000004 },
  { -5.5999999999999996, 1, -2.7999999999999998 },
  { -5.4000000000000004, 1, -2.7000000000000002 },
  { -5.1999999999999993, 1, -2.5999999999999996 },
  { -5.0000000000000000, 1, -2.5000000000000000 },
  { -4.8000000000000007, 1, -2.4000000000000004 },
  { -4.5999999999999996, 1, -2.2999999999999998 },
  { -4.4000000000000004, 1, -2.2000000000000002 },
  { -4.1999999999999993, 1, -2.0999999999999996 },
  { -4.0000000000000000, 1, -2.0000000000000000 },
  { -3.8000000000000007, 1, -1.9000000000000004 },
  { -3.6000000000000014, 1, -1.8000000000000007 },
  { -3.3999999999999986, 1, -1.6999999999999993 },
  { -3.1999999999999993, 1, -1.5999999999999996 },
  { -3.0000000000000000, 1, -1.5000000000000000 },
  { -2.8000000000000007, 1, -1.4000000000000004 },
  { -2.6000000000000014, 1, -1.3000000000000007 },
  { -2.3999999999999986, 1, -1.1999999999999993 },
  { -2.1999999999999993, 1, -1.0999999999999996 },
  { -2.0000000000000000, 1, -1.0000000000000000 },
  { -1.8000000000000007, 1, -0.90000000000000036 },
  { -1.6000000000000014, 1, -0.80000000000000071 },
  { -1.3999999999999986, 1, -0.69999999999999929 },
  { -1.1999999999999993, 1, -0.59999999999999964 },
  { -1.0000000000000000, 1, -0.50000000000000000 },
  { -0.80000000000000071, 1, -0.40000000000000036 },
  { -0.60000000000000142, 1, -0.30000000000000071 },
  { -0.39999999999999858, 1, -0.19999999999999929 },
  { -0.19999999999999929, 1, -0.099999999999999645 },
  { 0.0000000000000000, 1, 0.0000000000000000 },
  { 0.19999999999999929, 1, 0.099999999999999645 },
  { 0.39999999999999858, 1, 0.19999999999999929 },
  { 0.60000000000000142, 1, 0.30000000000000071 },
  { 0.80000000000000071, 1, 0.40000000000000036 },
  { 1.0000000000000000, 1, 0.50000000000000000 },
  { 1.1999999999999993, 1, 0.59999999999999964 },
  { 1.3999999999999986, 1, 0.69999999999999929 },
  { 1.6000000000000014, 1, 0.80000000000000071 },
  { 1.8000000000000007, 1, 0.90000000000000036 },
  { 2.0000000000000000, 1, 1.0000000000000000 },
  { 2.1999999999999993, 1, 1.0999999999999996 },
  { 2.3999999999999986, 1, 1.1999999999999993 },
  { 2.6000000000000014, 1, 1.3000000000000007 },
  { 2.8000000000000007, 1, 1.4000000000000004 },
  { 3.0000000000000000, 1, 1.5000000000000000 },
  { 3.1999999999999993, 1, 1.5999999999999996 },
  { 3.3999999999999986, 1, 1.6999999999999993 },
  { 3.6000000000000014, 1, 1.8000000000000007 },
  { 3.8000000000000007, 1, 1.9000000000000004 },
  { 4.0000000000000000, 1, 2.0000000000000000 },
  { 4.1999999999999993, 1, 2.0999999999999996 },
  { 4.3999999999999986, 1, 2.1999999999999993 },
  { 4.6000000000000014, 1, 2.3000000000000007 },
  { 4.8000000000000007, 1, 2.4000000000000004 },
  { 5.0000000000000000, 1, 2.5000000000000000 },
  { 5.1999999999999993, 1, 2.5999999999999996 },
  { 5.3999999999999986, 1, 2.6999999999999993 },
  { 5.6000000000000014, 1, 2.8000000000000007 },
  { 5.8000000000000007, 1, 2.9000000000000004 },
  { 6.0000000000000000, 1, 3.0000000000000000 },
  { 6.1999999999999993, 1, 3.0999999999999996 },
  { 6.3999999999999986, 1, 3.1999999999999993 },
  { 6.6000000000000014, 1, 3.3000000000000007 },
  { 6.8000000000000007, 1, 3.4000000000000004 },
  { 7.0000000000000000, 1, 3.5000000000000000 },
  { 7.1999999999999993, 1, 3.5999999999999996 },
  { 7.3999999999999986, 1, 3.6999999999999993 },
  { 7.6000000000000014, 1, 3.8000000000000007 },
  { 7.8000000000000007, 1, 3.9000000000000004 },
  { 8.0000000000000000, 1, 4.0000000000000000 },
  { 8.1999999999999993, 1, 4.0999999999999996 },
  { 8.3999999999999986, 1, 4.1999999999999993 },
  { 8.6000000000000014, 1, 4.3000000000000007 },
  { 8.8000000000000007, 1, 4.4000000000000004 },
  { 9.0000000000000000, 1, 4.5000000000000000 },
  { 9.1999999999999993, 1, 4.5999999999999996 },
  { 9.3999999999999986, 1, 4.6999999999999993 },
  { 9.6000000000000014, 1, 4.8000000000000007 },
  { 9.8000000000000007, 1, 4.9000000000000004 },
  { 10.000000000000000, 1, 5.0000000000000000 },
  { 10.199999999999999, 1, 5.0999999999999996 },
  { 10.399999999999999, 1, 5.1999999999999993 },
  { 10.600000000000001, 1, 5.3000000000000007 },
  { 10.800000000000001, 1, 5.4000000000000004 },
  { 11.000000000000000, 1, 5.5000000000000000 },
  { 11.199999999999999, 1, 5.5999999999999996 },
  { 11.399999999999999, 1, 5.6999999999999993 },
  { 11.600000000000001, 1, 5.8000000000000007 },
  { 11.800000000000001, 1, 5.9000000000000004 },
  { 12.000000000000000, 1, 6.0000000000000000 },
  { 12.200000000000003, 1, 6.1000000000000014 },
  { 12.399999999999999, 1, 6.1999999999999993 },
  { 12.600000000000001, 1, 6.3000000000000007 },
  { 12.799999999999997, 1, 6.3999999999999986 },
  { 13.000000000000000, 1, 6.5000000000000000 },
  { 13.200000000000003, 1, 6.6000000000000014 },
  { 13.399999999999999, 1, 6.6999999999999993 },
  { 13.600000000000001, 1, 6.8000000000000007 },
  { 13.799999999999997, 1, 6.8999999999999986 },
  { 14.000000000000000, 1, 7.0000000000000000 },
  { 14.200000000000003, 1, 7.1000000000000014 },
  { 14.399999999999999, 1, 7.1999999999999993 },
  { 14.600000000000001, 1, 7.3000000000000007 },
  { 14.799999999999997, 1, 7.3999999999999986 },
  { 15.000000000000000, 1, 7.5000000000000000 },
  { 15.200000000000003, 1, 7.6000000000000014 },
  { 15.399999999999999, 1, 7.6999999999999993 },
  { 15.600000000000001, 1, 7.8000000000000007 },
  { 15.799999999999997, 1, 7.8999999999999986 },
  { 16.000000000000000, 1, 8.0000000000000000 },
  { 16.200000000000003, 1, 8.1000000000000014 },
  { 16.399999999999999, 1, 8.1999999999999993 },
  { 16.600000000000001, 1, 8.3000000000000007 },
  { 16.799999999999997, 1, 8.3999999999999986 },
  { 17.000000000000000, 1, 8.5000000000000000 },
  { 17.200000000000003, 1, 8.6000000000000014 },
  { 17.399999999999999, 1, 8.6999999999999993 },
  { 17.600000000000001, 1, 8.8000000000000007 },
  { 17.799999999999997, 1, 8.8999999999999986 },
  { 18.000000000000000, 1, 9.0000000000000000 },
  { 18.200000000000003, 1, 9.1000000000000014 },
  { 18.399999999999999, 1, 9.1999999999999993 },
  { 18.600000000000001, 1, 9.3000000000000007 },
  { 18.799999999999997, 1, 9.3999999999999986 },
  { 19.000000000000000, 1, 9.5000000000000000 },
  { 19.200000000000003, 1, 9.6000000000000014 },
  { 19.399999999999999, 1, 9.6999999999999993 },
  { 19.600000000000001, 1, 9.8000000000000007 },
  { 19.799999999999997, 1, 9.8999999999999986 },
  { 20.000000000000000, 1, 10.000000000000000 },
};

// Test function for n=1.
template<typename Tp>
  void
  test002()
  {
    bool test __attribute__((unused)) = true;
    const Tp eps = std::numeric_limits<Tp>::epsilon();
    Tp max_abs_diff = -Tp(1);
    Tp max_abs_frac = -Tp(1);
    unsigned int num_datum = sizeof(data002)
			   / sizeof(testcase_hermite<double>);
    for (unsigned int i = 0; i < num_datum; ++i)
      {
	const Tp f = std::tr1::hermite(Tp(data002[i].n), Tp(data002[i].x));
	const Tp f0 = data002[i].f0;
	const Tp diff = f - f0;
	if (std::abs(diff) > max_abs_diff)
	  max_abs_diff = std::abs(diff);
	if (std::abs(f0) > Tp(10) * eps
	 && std::abs(f) > Tp(10) * eps)
	  {
	    const Tp frac = diff / f0;
	    if (std::abs(frac) > max_abs_frac)
	      max_abs_frac = std::abs(frac);
	  }
      }
    VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
  }

// Test data for n=2.
// max(|f - f_GSL|): 0.0000000000000000
// max(|f - f_GSL| / |f_GSL|): 0.0000000000000000
testcase_hermite<double>
data003[] =
{
  { 398.00000000000000, 2, -10.000000000000000 },
  { 390.04000000000002, 2, -9.9000000000000004 },
  { 382.16000000000008, 2, -9.8000000000000007 },
  { 374.35999999999996, 2, -9.6999999999999993 },
  { 366.63999999999999, 2, -9.5999999999999996 },
  { 359.00000000000000, 2, -9.5000000000000000 },
  { 351.44000000000005, 2, -9.4000000000000004 },
  { 343.96000000000004, 2, -9.3000000000000007 },
  { 336.55999999999995, 2, -9.1999999999999993 },
  { 329.23999999999995, 2, -9.0999999999999996 },
  { 322.00000000000000, 2, -9.0000000000000000 },
  { 314.84000000000003, 2, -8.9000000000000004 },
  { 307.76000000000005, 2, -8.8000000000000007 },
  { 300.75999999999993, 2, -8.6999999999999993 },
  { 293.83999999999997, 2, -8.5999999999999996 },
  { 287.00000000000000, 2, -8.5000000000000000 },
  { 280.24000000000001, 2, -8.4000000000000004 },
  { 273.56000000000006, 2, -8.3000000000000007 },
  { 266.95999999999998, 2, -8.1999999999999993 },
  { 260.44000000000000, 2, -8.0999999999999996 },
  { 254.00000000000000, 2, -8.0000000000000000 },
  { 247.64000000000001, 2, -7.9000000000000004 },
  { 241.35999999999999, 2, -7.7999999999999998 },
  { 235.16000000000003, 2, -7.7000000000000002 },
  { 229.03999999999999, 2, -7.5999999999999996 },
  { 223.00000000000000, 2, -7.5000000000000000 },
  { 217.04000000000002, 2, -7.4000000000000004 },
  { 211.16000000000000, 2, -7.2999999999999998 },
  { 205.36000000000001, 2, -7.2000000000000002 },
  { 199.63999999999999, 2, -7.0999999999999996 },
  { 194.00000000000000, 2, -7.0000000000000000 },
  { 188.44000000000003, 2, -6.9000000000000004 },
  { 182.95999999999998, 2, -6.7999999999999998 },
  { 177.56000000000000, 2, -6.7000000000000002 },
  { 172.23999999999998, 2, -6.5999999999999996 },
  { 167.00000000000000, 2, -6.5000000000000000 },
  { 161.84000000000003, 2, -6.4000000000000004 },
  { 156.75999999999999, 2, -6.2999999999999998 },
  { 151.76000000000002, 2, -6.2000000000000002 },
  { 146.83999999999997, 2, -6.0999999999999996 },
  { 142.00000000000000, 2, -6.0000000000000000 },
  { 137.24000000000001, 2, -5.9000000000000004 },
  { 132.56000000000000, 2, -5.7999999999999998 },
  { 127.96000000000001, 2, -5.7000000000000002 },
  { 123.43999999999998, 2, -5.5999999999999996 },
  { 119.00000000000000, 2, -5.5000000000000000 },
  { 114.64000000000001, 2, -5.4000000000000004 },
  { 110.36000000000000, 2, -5.2999999999999998 },
  { 106.16000000000001, 2, -5.2000000000000002 },
  { 102.03999999999999, 2, -5.0999999999999996 },
  { 98.000000000000000, 2, -5.0000000000000000 },
  { 94.040000000000020, 2, -4.9000000000000004 },
  { 90.159999999999997, 2, -4.7999999999999998 },
  { 86.360000000000014, 2, -4.7000000000000002 },
  { 82.639999999999986, 2, -4.5999999999999996 },
  { 79.000000000000000, 2, -4.5000000000000000 },
  { 75.440000000000012, 2, -4.4000000000000004 },
  { 71.959999999999994, 2, -4.2999999999999998 },
  { 68.560000000000002, 2, -4.2000000000000002 },
  { 65.239999999999995, 2, -4.0999999999999996 },
  { 62.000000000000000, 2, -4.0000000000000000 },
  { 58.840000000000011, 2, -3.9000000000000004 },
  { 55.759999999999998, 2, -3.7999999999999998 },
  { 52.760000000000005, 2, -3.7000000000000002 },
  { 49.839999999999989, 2, -3.5999999999999996 },
  { 47.000000000000000, 2, -3.5000000000000000 },
  { 44.240000000000009, 2, -3.4000000000000004 },
  { 41.559999999999995, 2, -3.2999999999999998 },
  { 38.960000000000008, 2, -3.2000000000000002 },
  { 36.439999999999991, 2, -3.0999999999999996 },
  { 34.000000000000000, 2, -3.0000000000000000 },
  { 31.640000000000008, 2, -2.9000000000000004 },
  { 29.359999999999996, 2, -2.7999999999999998 },
  { 27.160000000000004, 2, -2.7000000000000002 },
  { 25.039999999999992, 2, -2.5999999999999996 },
  { 23.000000000000000, 2, -2.5000000000000000 },
  { 21.040000000000006, 2, -2.4000000000000004 },
  { 19.159999999999997, 2, -2.2999999999999998 },
  { 17.360000000000003, 2, -2.2000000000000002 },
  { 15.639999999999993, 2, -2.0999999999999996 },
  { 14.000000000000000, 2, -2.0000000000000000 },
  { 12.440000000000005, 2, -1.9000000000000004 },
  { 10.960000000000010, 2, -1.8000000000000007 },
  { 9.5599999999999898, 2, -1.6999999999999993 },
  { 8.2399999999999949, 2, -1.5999999999999996 },
  { 7.0000000000000000, 2, -1.5000000000000000 },
  { 5.8400000000000043, 2, -1.4000000000000004 },
  { 4.7600000000000078, 2, -1.3000000000000007 },
  { 3.7599999999999936, 2, -1.1999999999999993 },
  { 2.8399999999999972, 2, -1.0999999999999996 },
  { 2.0000000000000000, 2, -1.0000000000000000 },
  { 1.2400000000000024, 2, -0.90000000000000036 },
  { 0.56000000000000449, 2, -0.80000000000000071 },
  { -0.040000000000004032, 2, -0.69999999999999929 },
  { -0.56000000000000161, 2, -0.59999999999999964 },
  { -1.0000000000000000, 2, -0.50000000000000000 },
  { -1.3599999999999990, 2, -0.40000000000000036 },
  { -1.6399999999999983, 2, -0.30000000000000071 },
  { -1.8400000000000012, 2, -0.19999999999999929 },
  { -1.9600000000000002, 2, -0.099999999999999645 },
  { -2.0000000000000000, 2, 0.0000000000000000 },
  { -1.9600000000000002, 2, 0.099999999999999645 },
  { -1.8400000000000012, 2, 0.19999999999999929 },
  { -1.6399999999999983, 2, 0.30000000000000071 },
  { -1.3599999999999990, 2, 0.40000000000000036 },
  { -1.0000000000000000, 2, 0.50000000000000000 },
  { -0.56000000000000161, 2, 0.59999999999999964 },
  { -0.040000000000004032, 2, 0.69999999999999929 },
  { 0.56000000000000449, 2, 0.80000000000000071 },
  { 1.2400000000000024, 2, 0.90000000000000036 },
  { 2.0000000000000000, 2, 1.0000000000000000 },
  { 2.8399999999999972, 2, 1.0999999999999996 },
  { 3.7599999999999936, 2, 1.1999999999999993 },
  { 4.7600000000000078, 2, 1.3000000000000007 },
  { 5.8400000000000043, 2, 1.4000000000000004 },
  { 7.0000000000000000, 2, 1.5000000000000000 },
  { 8.2399999999999949, 2, 1.5999999999999996 },
  { 9.5599999999999898, 2, 1.6999999999999993 },
  { 10.960000000000010, 2, 1.8000000000000007 },
  { 12.440000000000005, 2, 1.9000000000000004 },
  { 14.000000000000000, 2, 2.0000000000000000 },
  { 15.639999999999993, 2, 2.0999999999999996 },
  { 17.359999999999989, 2, 2.1999999999999993 },
  { 19.160000000000014, 2, 2.3000000000000007 },
  { 21.040000000000006, 2, 2.4000000000000004 },
  { 23.000000000000000, 2, 2.5000000000000000 },
  { 25.039999999999992, 2, 2.5999999999999996 },
  { 27.159999999999986, 2, 2.6999999999999993 },
  { 29.360000000000017, 2, 2.8000000000000007 },
  { 31.640000000000008, 2, 2.9000000000000004 },
  { 34.000000000000000, 2, 3.0000000000000000 },
  { 36.439999999999991, 2, 3.0999999999999996 },
  { 38.959999999999980, 2, 3.1999999999999993 },
  { 41.560000000000016, 2, 3.3000000000000007 },
  { 44.240000000000009, 2, 3.4000000000000004 },
  { 47.000000000000000, 2, 3.5000000000000000 },
  { 49.839999999999989, 2, 3.5999999999999996 },
  { 52.759999999999977, 2, 3.6999999999999993 },
  { 55.760000000000019, 2, 3.8000000000000007 },
  { 58.840000000000011, 2, 3.9000000000000004 },
  { 62.000000000000000, 2, 4.0000000000000000 },
  { 65.239999999999995, 2, 4.0999999999999996 },
  { 68.559999999999974, 2, 4.1999999999999993 },
  { 71.960000000000022, 2, 4.3000000000000007 },
  { 75.440000000000012, 2, 4.4000000000000004 },
  { 79.000000000000000, 2, 4.5000000000000000 },
  { 82.639999999999986, 2, 4.5999999999999996 },
  { 86.359999999999971, 2, 4.6999999999999993 },
  { 90.160000000000025, 2, 4.8000000000000007 },
  { 94.040000000000020, 2, 4.9000000000000004 },
  { 98.000000000000000, 2, 5.0000000000000000 },
  { 102.03999999999999, 2, 5.0999999999999996 },
  { 106.15999999999997, 2, 5.1999999999999993 },
  { 110.36000000000003, 2, 5.3000000000000007 },
  { 114.64000000000001, 2, 5.4000000000000004 },
  { 119.00000000000000, 2, 5.5000000000000000 },
  { 123.43999999999998, 2, 5.5999999999999996 },
  { 127.95999999999998, 2, 5.6999999999999993 },
  { 132.56000000000003, 2, 5.8000000000000007 },
  { 137.24000000000001, 2, 5.9000000000000004 },
  { 142.00000000000000, 2, 6.0000000000000000 },
  { 146.84000000000006, 2, 6.1000000000000014 },
  { 151.75999999999996, 2, 6.1999999999999993 },
  { 156.76000000000005, 2, 6.3000000000000007 },
  { 161.83999999999992, 2, 6.3999999999999986 },
  { 167.00000000000000, 2, 6.5000000000000000 },
  { 172.24000000000007, 2, 6.6000000000000014 },
  { 177.55999999999997, 2, 6.6999999999999993 },
  { 182.96000000000004, 2, 6.8000000000000007 },
  { 188.43999999999991, 2, 6.8999999999999986 },
  { 194.00000000000000, 2, 7.0000000000000000 },
  { 199.64000000000007, 2, 7.1000000000000014 },
  { 205.35999999999996, 2, 7.1999999999999993 },
  { 211.16000000000005, 2, 7.3000000000000007 },
  { 217.03999999999991, 2, 7.3999999999999986 },
  { 223.00000000000000, 2, 7.5000000000000000 },
  { 229.04000000000008, 2, 7.6000000000000014 },
  { 235.15999999999997, 2, 7.6999999999999993 },
  { 241.36000000000004, 2, 7.8000000000000007 },
  { 247.63999999999990, 2, 7.8999999999999986 },
  { 254.00000000000000, 2, 8.0000000000000000 },
  { 260.44000000000011, 2, 8.1000000000000014 },
  { 266.95999999999998, 2, 8.1999999999999993 },
  { 273.56000000000006, 2, 8.3000000000000007 },
  { 280.23999999999990, 2, 8.3999999999999986 },
  { 287.00000000000000, 2, 8.5000000000000000 },
  { 293.84000000000009, 2, 8.6000000000000014 },
  { 300.75999999999993, 2, 8.6999999999999993 },
  { 307.76000000000005, 2, 8.8000000000000007 },
  { 314.83999999999992, 2, 8.8999999999999986 },
  { 322.00000000000000, 2, 9.0000000000000000 },
  { 329.24000000000012, 2, 9.1000000000000014 },
  { 336.55999999999995, 2, 9.1999999999999993 },
  { 343.96000000000004, 2, 9.3000000000000007 },
  { 351.43999999999988, 2, 9.3999999999999986 },
  { 359.00000000000000, 2, 9.5000000000000000 },
  { 366.64000000000010, 2, 9.6000000000000014 },
  { 374.35999999999996, 2, 9.6999999999999993 },
  { 382.16000000000008, 2, 9.8000000000000007 },
  { 390.03999999999991, 2, 9.8999999999999986 },
  { 398.00000000000000, 2, 10.000000000000000 },
};

// Test function for n=2.
template<typename Tp>
  void
  test003()
  {
    bool test __attribute__((unused)) = true;
    const Tp eps = std::numeric_limits<Tp>::epsilon();
    Tp max_abs_diff = -Tp(1);
    Tp max_abs_frac = -Tp(1);
    unsigned int num_datum = sizeof(data003)
			   / sizeof(testcase_hermite<double>);
    for (unsigned int i = 0; i < num_datum; ++i)
      {
	const Tp f = std::tr1::hermite(Tp(data003[i].n), Tp(data003[i].x));
	const Tp f0 = data003[i].f0;
	const Tp diff = f - f0;
	if (std::abs(diff) > max_abs_diff)
	  max_abs_diff = std::abs(diff);
	if (std::abs(f0) > Tp(10) * eps
	 && std::abs(f) > Tp(10) * eps)
	  {
	    const Tp frac = diff / f0;
	    if (std::abs(frac) > max_abs_frac)
	      max_abs_frac = std::abs(frac);
	  }
      }
    VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
  }

// Test data for n=5.
// max(|f - f_GSL|): 0.0000000000000000
// max(|f - f_GSL| / |f_GSL|): 0.0000000000000000
testcase_hermite<double>
data004[] =
{
  { -3041200.0000000000, 5, -10.000000000000000 },
  { -2889108.3196800007, 5, -9.9000000000000004 },
  { -2743131.8297600015, 5, -9.8000000000000007 },
  { -2603085.2022399991, 5, -9.6999999999999993 },
  { -2468786.8723199992, 5, -9.5999999999999996 },
  { -2340059.0000000000, 5, -9.5000000000000000 },
  { -2216727.4316800009, 5, -9.4000000000000004 },
  { -2098621.6617600010, 5, -9.3000000000000007 },
  { -1985574.7942399993, 5, -9.1999999999999993 },
  { -1877423.5043199996, 5, -9.0999999999999996 },
  { -1774008.0000000000, 5, -9.0000000000000000 },
  { -1675171.9836800008, 5, -8.9000000000000004 },
  { -1580762.6137600006, 5, -8.8000000000000007 },
  { -1490630.4662399990, 5, -8.6999999999999993 },
  { -1404629.4963199999, 5, -8.5999999999999996 },
  { -1322617.0000000000, 5, -8.5000000000000000 },
  { -1244453.5756800002, 5, -8.4000000000000004 },
  { -1170003.0857600006, 5, -8.3000000000000007 },
  { -1099132.6182399995, 5, -8.1999999999999993 },
  { -1031712.4483199996, 5, -8.0999999999999996 },
  { -967616.00000000000, 5, -8.0000000000000000 },
  { -906719.80768000009, 5, -7.9000000000000004 },
  { -848903.47775999992, 5, -7.7999999999999998 },
  { -794049.65024000010, 5, -7.7000000000000002 },
  { -742043.96031999995, 5, -7.5999999999999996 },
  { -692775.00000000000, 5, -7.5000000000000000 },
  { -646134.27968000027, 5, -7.4000000000000004 },
  { -602016.18975999986, 5, -7.2999999999999998 },
  { -560317.96224000002, 5, -7.2000000000000002 },
  { -520939.63231999992, 5, -7.0999999999999996 },
  { -483784.00000000000, 5, -7.0000000000000000 },
  { -448756.59168000019, 5, -6.9000000000000004 },
  { -415765.62176000001, 5, -6.7999999999999998 },
  { -384721.95424000005, 5, -6.7000000000000002 },
  { -355539.06431999995, 5, -6.5999999999999996 },
  { -328133.00000000000, 5, -6.5000000000000000 },
  { -302422.34368000011, 5, -6.4000000000000004 },
  { -278328.17375999998, 5, -6.2999999999999998 },
  { -255774.02624000004, 5, -6.2000000000000002 },
  { -234685.85631999990, 5, -6.0999999999999996 },
  { -214992.00000000000, 5, -6.0000000000000000 },
  { -196623.13568000006, 5, -5.9000000000000004 },
  { -179512.24575999999, 5, -5.7999999999999998 },
  { -163594.57824000003, 5, -5.7000000000000002 },
  { -148807.60831999997, 5, -5.5999999999999996 },
  { -135091.00000000000, 5, -5.5000000000000000 },
  { -122386.56768000004, 5, -5.4000000000000004 },
  { -110638.23776000000, 5, -5.2999999999999998 },
  { -99792.010240000047, 5, -5.2000000000000002 },
  { -89795.920319999976, 5, -5.0999999999999996 },
  { -80600.000000000000, 5, -5.0000000000000000 },
  { -72156.239680000028, 5, -4.9000000000000004 },
  { -64418.549759999994, 5, -4.7999999999999998 },
  { -57342.722240000010, 5, -4.7000000000000002 },
  { -50886.392319999970, 5, -4.5999999999999996 },
  { -45009.000000000000, 5, -4.5000000000000000 },
  { -39671.751680000008, 5, -4.4000000000000004 },
  { -34837.581759999986, 5, -4.2999999999999998 },
  { -30471.114240000010, 5, -4.2000000000000002 },
  { -26538.624319999992, 5, -4.0999999999999996 },
  { -23008.000000000000, 5, -4.0000000000000000 },
  { -19848.703680000013, 5, -3.9000000000000004 },
  { -17031.733759999996, 5, -3.7999999999999998 },
  { -14529.586240000001, 5, -3.7000000000000002 },
  { -12316.216319999992, 5, -3.5999999999999996 },
  { -10367.000000000000, 5, -3.5000000000000000 },
  { -8658.6956800000080, 5, -3.4000000000000004 },
  { -7169.4057599999960, 5, -3.2999999999999998 },
  { -5878.5382400000035, 5, -3.2000000000000002 },
  { -4766.7683199999965, 5, -3.0999999999999996 },
  { -3816.0000000000000, 5, -3.0000000000000000 },
  { -3009.3276800000026, 5, -2.9000000000000004 },
  { -2330.9977599999988, 5, -2.7999999999999998 },
  { -1766.3702400000002, 5, -2.7000000000000002 },
  { -1301.8803199999984, 5, -2.5999999999999996 },
  { -925.00000000000000, 5, -2.5000000000000000 },
  { -624.19968000000074, 5, -2.4000000000000004 },
  { -388.90975999999978, 5, -2.2999999999999998 },
  { -209.48224000000022, 5, -2.2000000000000002 },
  { -77.152319999999520, 5, -2.0999999999999996 },
  { 16.000000000000000, 5, -2.0000000000000000 },
  { 77.088319999999840, 5, -1.9000000000000004 },
  { 112.45823999999980, 5, -1.8000000000000007 },
  { 127.72576000000002, 5, -1.6999999999999993 },
  { 127.81567999999996, 5, -1.5999999999999996 },
  { 117.00000000000000, 5, -1.5000000000000000 },
  { 98.936320000000080, 5, -1.4000000000000004 },
  { 76.706240000000179, 5, -1.3000000000000007 },
  { 52.853759999999838, 5, -1.1999999999999993 },
  { 29.423679999999933, 5, -1.0999999999999996 },
  { 8.0000000000000000, 5, -1.0000000000000000 },
  { -10.255679999999948, 5, -0.90000000000000036 },
  { -24.565759999999916, 5, -0.80000000000000071 },
  { -34.498240000000052, 5, -0.69999999999999929 },
  { -39.928320000000006, 5, -0.59999999999999964 },
  { -41.000000000000000, 5, -0.50000000000000000 },
  { -38.087680000000020, 5, -0.40000000000000036 },
  { -31.757760000000054, 5, -0.30000000000000071 },
  { -22.730239999999931, 5, -0.19999999999999929 },
  { -11.840319999999959, 5, -0.099999999999999645 },
  { 0.0000000000000000, 5, 0.0000000000000000 },
  { 11.840319999999959, 5, 0.099999999999999645 },
  { 22.730239999999931, 5, 0.19999999999999929 },
  { 31.757760000000054, 5, 0.30000000000000071 },
  { 38.087680000000020, 5, 0.40000000000000036 },
  { 41.000000000000000, 5, 0.50000000000000000 },
  { 39.928320000000006, 5, 0.59999999999999964 },
  { 34.498240000000052, 5, 0.69999999999999929 },
  { 24.565759999999916, 5, 0.80000000000000071 },
  { 10.255679999999948, 5, 0.90000000000000036 },
  { -8.0000000000000000, 5, 1.0000000000000000 },
  { -29.423679999999933, 5, 1.0999999999999996 },
  { -52.853759999999838, 5, 1.1999999999999993 },
  { -76.706240000000179, 5, 1.3000000000000007 },
  { -98.936320000000080, 5, 1.4000000000000004 },
  { -117.00000000000000, 5, 1.5000000000000000 },
  { -127.81567999999996, 5, 1.5999999999999996 },
  { -127.72576000000002, 5, 1.6999999999999993 },
  { -112.45823999999980, 5, 1.8000000000000007 },
  { -77.088319999999840, 5, 1.9000000000000004 },
  { -16.000000000000000, 5, 2.0000000000000000 },
  { 77.152319999999520, 5, 2.0999999999999996 },
  { 209.48223999999891, 5, 2.1999999999999993 },
  { 388.90976000000148, 5, 2.3000000000000007 },
  { 624.19968000000074, 5, 2.4000000000000004 },
  { 925.00000000000000, 5, 2.5000000000000000 },
  { 1301.8803199999984, 5, 2.5999999999999996 },
  { 1766.3702399999963, 5, 2.6999999999999993 },
  { 2330.9977600000047, 5, 2.8000000000000007 },
  { 3009.3276800000026, 5, 2.9000000000000004 },
  { 3816.0000000000000, 5, 3.0000000000000000 },
  { 4766.7683199999965, 5, 3.0999999999999996 },
  { 5878.5382399999908, 5, 3.1999999999999993 },
  { 7169.4057600000087, 5, 3.3000000000000007 },
  { 8658.6956800000080, 5, 3.4000000000000004 },
  { 10367.000000000000, 5, 3.5000000000000000 },
  { 12316.216319999992, 5, 3.5999999999999996 },
  { 14529.586239999979, 5, 3.6999999999999993 },
  { 17031.733760000021, 5, 3.8000000000000007 },
  { 19848.703680000013, 5, 3.9000000000000004 },
  { 23008.000000000000, 5, 4.0000000000000000 },
  { 26538.624319999992, 5, 4.0999999999999996 },
  { 30471.114239999963, 5, 4.1999999999999993 },
  { 34837.581760000037, 5, 4.3000000000000007 },
  { 39671.751680000008, 5, 4.4000000000000004 },
  { 45009.000000000000, 5, 4.5000000000000000 },
  { 50886.392319999970, 5, 4.5999999999999996 },
  { 57342.722239999952, 5, 4.6999999999999993 },
  { 64418.549760000053, 5, 4.8000000000000007 },
  { 72156.239680000028, 5, 4.9000000000000004 },
  { 80600.000000000000, 5, 5.0000000000000000 },
  { 89795.920319999976, 5, 5.0999999999999996 },
  { 99792.010239999945, 5, 5.1999999999999993 },
  { 110638.23776000006, 5, 5.3000000000000007 },
  { 122386.56768000004, 5, 5.4000000000000004 },
  { 135091.00000000000, 5, 5.5000000000000000 },
  { 148807.60831999997, 5, 5.5999999999999996 },
  { 163594.57823999992, 5, 5.6999999999999993 },
  { 179512.24576000011, 5, 5.8000000000000007 },
  { 196623.13568000006, 5, 5.9000000000000004 },
  { 214992.00000000000, 5, 6.0000000000000000 },
  { 234685.85632000031, 5, 6.1000000000000014 },
  { 255774.02623999983, 5, 6.1999999999999993 },
  { 278328.17376000021, 5, 6.3000000000000007 },
  { 302422.34367999958, 5, 6.3999999999999986 },
  { 328133.00000000000, 5, 6.5000000000000000 },
  { 355539.06432000035, 5, 6.6000000000000014 },
  { 384721.95423999976, 5, 6.6999999999999993 },
  { 415765.62176000018, 5, 6.8000000000000007 },
  { 448756.59167999960, 5, 6.8999999999999986 },
  { 483784.00000000000, 5, 7.0000000000000000 },
  { 520939.63232000044, 5, 7.1000000000000014 },
  { 560317.96223999979, 5, 7.1999999999999993 },
  { 602016.18976000033, 5, 7.3000000000000007 },
  { 646134.27967999945, 5, 7.3999999999999986 },
  { 692775.00000000000, 5, 7.5000000000000000 },
  { 742043.96032000054, 5, 7.6000000000000014 },
  { 794049.65023999964, 5, 7.6999999999999993 },
  { 848903.47776000027, 5, 7.8000000000000007 },
  { 906719.80767999915, 5, 7.8999999999999986 },
  { 967616.00000000000, 5, 8.0000000000000000 },
  { 1031712.4483200011, 5, 8.1000000000000014 },
  { 1099132.6182399995, 5, 8.1999999999999993 },
  { 1170003.0857600006, 5, 8.3000000000000007 },
  { 1244453.5756799988, 5, 8.3999999999999986 },
  { 1322617.0000000000, 5, 8.5000000000000000 },
  { 1404629.4963200013, 5, 8.6000000000000014 },
  { 1490630.4662399990, 5, 8.6999999999999993 },
  { 1580762.6137600006, 5, 8.8000000000000007 },
  { 1675171.9836799989, 5, 8.8999999999999986 },
  { 1774008.0000000000, 5, 9.0000000000000000 },
  { 1877423.5043200015, 5, 9.1000000000000014 },
  { 1985574.7942399993, 5, 9.1999999999999993 },
  { 2098621.6617600010, 5, 9.3000000000000007 },
  { 2216727.4316799981, 5, 9.3999999999999986 },
  { 2340059.0000000000, 5, 9.5000000000000000 },
  { 2468786.8723200019, 5, 9.6000000000000014 },
  { 2603085.2022399991, 5, 9.6999999999999993 },
  { 2743131.8297600015, 5, 9.8000000000000007 },
  { 2889108.3196799983, 5, 9.8999999999999986 },
  { 3041200.0000000000, 5, 10.000000000000000 },
};

// Test function for n=5.
template<typename Tp>
  void
  test004()
  {
    bool test __attribute__((unused)) = true;
    const Tp eps = std::numeric_limits<Tp>::epsilon();
    Tp max_abs_diff = -Tp(1);
    Tp max_abs_frac = -Tp(1);
    unsigned int num_datum = sizeof(data004)
			   / sizeof(testcase_hermite<double>);
    for (unsigned int i = 0; i < num_datum; ++i)
      {
	const Tp f = std::tr1::hermite(Tp(data004[i].n), Tp(data004[i].x));
	const Tp f0 = data004[i].f0;
	const Tp diff = f - f0;
	if (std::abs(diff) > max_abs_diff)
	  max_abs_diff = std::abs(diff);
	if (std::abs(f0) > Tp(10) * eps
	 && std::abs(f) > Tp(10) * eps)
	  {
	    const Tp frac = diff / f0;
	    if (std::abs(frac) > max_abs_frac)
	      max_abs_frac = std::abs(frac);
	  }
      }
    VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
  }

// Test data for n=10.
// max(|f - f_GSL|): 0.0000000000000000
// max(|f - f_GSL| / |f_GSL|): 0.0000000000000000
testcase_hermite<double>
data005[] =
{
  { 8093278209760.0000, 10, -10.000000000000000 },
  { 7282867075495.3066, 10, -9.9000000000000004 },
  { 6545846221520.6768, 10, -9.8000000000000007 },
  { 5876279006180.6377, 10, -9.6999999999999993 },
  { 5268651052510.4668, 10, -9.5999999999999996 },
  { 4717844376391.0000, 10, -9.5000000000000000 },
  { 4219112842239.1147, 10, -9.4000000000000004 },
  { 3768058891466.0469, 10, -9.3000000000000007 },
  { 3360611490639.0889, 10, -9.1999999999999993 },
  { 2993005247949.7607, 10, -9.0999999999999996 },
  { 2661760648224.0000, 10, -9.0000000000000000 },
  { 2363665358307.8496, 10, -8.9000000000000004 },
  { 2095756556225.3428, 10, -8.8000000000000007 },
  { 1855304239034.7983, 10, -8.6999999999999993 },
  { 1639795465805.4746, 10, -8.5999999999999996 },
  { 1446919493599.0000, 10, -8.5000000000000000 },
  { 1274553765769.7463, 10, -8.4000000000000004 },
  { 1120750713295.2778, 10, -8.3000000000000007 },
  { 983725331213.07190, 10, -8.1999999999999993 },
  { 861843493572.90051, 10, -8.0999999999999996 },
  { 753610971616.00000, 10, -8.0000000000000000 },
  { 657663121163.02454, 10, -7.9000000000000004 },
  { 572755206432.81335, 10, -7.7999999999999998 },
  { 497753328723.87518, 10, -7.7000000000000002 },
  { 431625929570.40063, 10, -7.5999999999999996 },
  { 373435839135.00000, 10, -7.5000000000000000 },
  { 322332841721.55731, 10, -7.4000000000000004 },
  { 277546731384.01782, 10, -7.2999999999999998 },
  { 238380831670.89990, 10, -7.2000000000000002 },
  { 204205954581.24731, 10, -7.0999999999999996 },
  { 174454774816.00000, 10, -7.0000000000000000 },
  { 148616596389.67230, 10, -6.9000000000000004 },
  { 126232489621.23923, 10, -6.7999999999999998 },
  { 106890777450.54663, 10, -6.7000000000000002 },
  { 90222850927.787979, 10, -6.5999999999999996 },
  { 75899294599.000000, 10, -6.5000000000000000 },
  { 63626303360.468109, 10, -6.4000000000000004 },
  { 53142373179.806053, 10, -6.2999999999999998 },
  { 44215248881.625534, 10, -6.2000000000000002 },
  { 36639112971.527679, 10, -6.0999999999999996 },
  { 30232000224.000000, 10, -6.0000000000000000 },
  { 24833423488.050617, 10, -5.9000000000000004 },
  { 20302196869.444584, 10, -5.7999999999999998 },
  { 16514443130.579359, 10, -5.7000000000000002 },
  { 13361772808.732477, 10, -5.5999999999999996 },
  { 10749623191.000000, 10, -5.5000000000000000 },
  { 8595745900.0898170, 10, -5.4000000000000004 },
  { 6828832439.6160822, 10, -5.2999999999999998 },
  { 5387267621.0259018, 10, -5.2000000000000002 },
  { 4218001347.1524582, 10, -5.0999999999999996 },
  { 3275529760.0000000, 10, -5.0000000000000000 },
  { 2520977273.0966806, 10, -4.9000000000000004 },
  { 1921271501.9744270, 10, -4.7999999999999998 },
  { 1448403580.4203825, 10, -4.7000000000000002 },
  { 1078766805.4651456, 10, -4.5999999999999996 },
  { 792566991.00000000, 10, -4.5000000000000000 },
  { 573298328.81993556, 10, -4.4000000000000004 },
  { 407278957.14375639, 10, -4.2999999999999998 },
  { 283240820.63788313, 10, -4.2000000000000002 },
  { 191968773.03860721, 10, -4.0999999999999996 },
  { 125984224.00000000, 10, -4.0000000000000000 },
  { 79268966.162877649, 10, -3.9000000000000004 },
  { 47025137.016035721, 10, -3.7999999999999998 },
  { 25467573.275709353, 10, -3.7000000000000002 },
  { 11645103.614666298, 10, -3.5999999999999996 },
  { 3287599.0000000000, 10, -3.5000000000000000 },
  { -1324140.9798373245, 10, -3.4000000000000004 },
  { -3468342.2313268245, 10, -3.2999999999999998 },
  { -4074495.5241857050, 10, -3.2000000000000002 },
  { -3800107.4878923763, 10, -3.0999999999999996 },
  { -3093984.0000000000, 10, -3.0000000000000000 },
  { -2247873.5653938209, 10, -2.9000000000000004 },
  { -1438117.1978829810, 10, -2.7999999999999998 },
  { -758781.93281034287, 10, -2.7000000000000002 },
  { -247597.05012469599, 10, -2.5999999999999996 },
  { 94135.000000000000, 10, -2.5000000000000000 },
  { 286617.47398410190, 10, -2.4000000000000004 },
  { 360718.79745525768, 10, -2.2999999999999998 },
  { 350419.82826741762, 10, -2.2000000000000002 },
  { 287863.09027338214, 10, -2.0999999999999996 },
  { 200416.00000000000, 10, -2.0000000000000000 },
  { 109249.22783242268, 10, -1.9000000000000004 },
  { 29012.094015898125, 10, -1.8000000000000007 },
  { -31740.330680422732, 10, -1.6999999999999993 },
  { -69648.597834137676, 10, -1.5999999999999996 },
  { -85401.000000000000, 10, -1.5000000000000000 },
  { -82507.675752857642, 10, -1.4000000000000004 },
  { -66123.413033062563, 10, -1.3000000000000007 },
  { -42007.465141862223, 10, -1.1999999999999993 },
  { -15676.055823257526, 10, -1.0999999999999996 },
  { 8224.0000000000000, 10, -1.0000000000000000 },
  { 26314.366684262357, 10, -0.90000000000000036 },
  { 36668.344916377559, 10, -0.80000000000000071 },
  { 38802.826035097583, 10, -0.69999999999999929 },
  { 33513.167890022363, 10, -0.59999999999999964 },
  { 22591.000000000000, 10, -0.50000000000000000 },
  { 8467.6907597824556, 10, -0.40000000000000036 },
  { -6173.8524877822965, 10, -0.30000000000000071 },
  { -18778.856957542470, 10, -0.19999999999999929 },
  { -27256.158950297624, 10, -0.099999999999999645 },
  { -30240.000000000000, 10, 0.0000000000000000 },
  { -27256.158950297624, 10, 0.099999999999999645 },
  { -18778.856957542470, 10, 0.19999999999999929 },
  { -6173.8524877822965, 10, 0.30000000000000071 },
  { 8467.6907597824556, 10, 0.40000000000000036 },
  { 22591.000000000000, 10, 0.50000000000000000 },
  { 33513.167890022363, 10, 0.59999999999999964 },
  { 38802.826035097583, 10, 0.69999999999999929 },
  { 36668.344916377559, 10, 0.80000000000000071 },
  { 26314.366684262357, 10, 0.90000000000000036 },
  { 8224.0000000000000, 10, 1.0000000000000000 },
  { -15676.055823257526, 10, 1.0999999999999996 },
  { -42007.465141862223, 10, 1.1999999999999993 },
  { -66123.413033062563, 10, 1.3000000000000007 },
  { -82507.675752857642, 10, 1.4000000000000004 },
  { -85401.000000000000, 10, 1.5000000000000000 },
  { -69648.597834137676, 10, 1.5999999999999996 },
  { -31740.330680422732, 10, 1.6999999999999993 },
  { 29012.094015898125, 10, 1.8000000000000007 },
  { 109249.22783242268, 10, 1.9000000000000004 },
  { 200416.00000000000, 10, 2.0000000000000000 },
  { 287863.09027338214, 10, 2.0999999999999996 },
  { 350419.82826741732, 10, 2.1999999999999993 },
  { 360718.79745525745, 10, 2.3000000000000007 },
  { 286617.47398410190, 10, 2.4000000000000004 },
  { 94135.000000000000, 10, 2.5000000000000000 },
  { -247597.05012469599, 10, 2.5999999999999996 },
  { -758781.93281033845, 10, 2.6999999999999993 },
  { -1438117.1978829878, 10, 2.8000000000000007 },
  { -2247873.5653938209, 10, 2.9000000000000004 },
  { -3093984.0000000000, 10, 3.0000000000000000 },
  { -3800107.4878923763, 10, 3.0999999999999996 },
  { -4074495.5241857003, 10, 3.1999999999999993 },
  { -3468342.2313268133, 10, 3.3000000000000007 },
  { -1324140.9798373245, 10, 3.4000000000000004 },
  { 3287599.0000000000, 10, 3.5000000000000000 },
  { 11645103.614666298, 10, 3.5999999999999996 },
  { 25467573.275709212, 10, 3.6999999999999993 },
  { 47025137.016035900, 10, 3.8000000000000007 },
  { 79268966.162877649, 10, 3.9000000000000004 },
  { 125984224.00000000, 10, 4.0000000000000000 },
  { 191968773.03860721, 10, 4.0999999999999996 },
  { 283240820.63788199, 10, 4.1999999999999993 },
  { 407278957.14375770, 10, 4.3000000000000007 },
  { 573298328.81993556, 10, 4.4000000000000004 },
  { 792566991.00000000, 10, 4.5000000000000000 },
  { 1078766805.4651456, 10, 4.5999999999999996 },
  { 1448403580.4203794, 10, 4.6999999999999993 },
  { 1921271501.9744320, 10, 4.8000000000000007 },
  { 2520977273.0966806, 10, 4.9000000000000004 },
  { 3275529760.0000000, 10, 5.0000000000000000 },
  { 4218001347.1524582, 10, 5.0999999999999996 },
  { 5387267621.0258913, 10, 5.1999999999999993 },
  { 6828832439.6160927, 10, 5.3000000000000007 },
  { 8595745900.0898170, 10, 5.4000000000000004 },
  { 10749623191.000000, 10, 5.5000000000000000 },
  { 13361772808.732477, 10, 5.5999999999999996 },
  { 16514443130.579332, 10, 5.6999999999999993 },
  { 20302196869.444618, 10, 5.8000000000000007 },
  { 24833423488.050617, 10, 5.9000000000000004 },
  { 30232000224.000000, 10, 6.0000000000000000 },
  { 36639112971.527824, 10, 6.1000000000000014 },
  { 44215248881.625443, 10, 6.1999999999999993 },
  { 53142373179.806152, 10, 6.3000000000000007 },
  { 63626303360.467911, 10, 6.3999999999999986 },
  { 75899294599.000000, 10, 6.5000000000000000 },
  { 90222850927.788208, 10, 6.6000000000000014 },
  { 106890777450.54652, 10, 6.6999999999999993 },
  { 126232489621.23946, 10, 6.8000000000000007 },
  { 148616596389.67184, 10, 6.8999999999999986 },
  { 174454774816.00000, 10, 7.0000000000000000 },
  { 204205954581.24780, 10, 7.1000000000000014 },
  { 238380831670.89960, 10, 7.1999999999999993 },
  { 277546731384.01831, 10, 7.3000000000000007 },
  { 322332841721.55646, 10, 7.3999999999999986 },
  { 373435839135.00000, 10, 7.5000000000000000 },
  { 431625929570.40161, 10, 7.6000000000000014 },
  { 497753328723.87476, 10, 7.6999999999999993 },
  { 572755206432.81396, 10, 7.8000000000000007 },
  { 657663121163.02307, 10, 7.8999999999999986 },
  { 753610971616.00000, 10, 8.0000000000000000 },
  { 861843493572.90283, 10, 8.1000000000000014 },
  { 983725331213.07190, 10, 8.1999999999999993 },
  { 1120750713295.2778, 10, 8.3000000000000007 },
  { 1274553765769.7439, 10, 8.3999999999999986 },
  { 1446919493599.0000, 10, 8.5000000000000000 },
  { 1639795465805.4785, 10, 8.6000000000000014 },
  { 1855304239034.7983, 10, 8.6999999999999993 },
  { 2095756556225.3428, 10, 8.8000000000000007 },
  { 2363665358307.8442, 10, 8.8999999999999986 },
  { 2661760648224.0000, 10, 9.0000000000000000 },
  { 2993005247949.7671, 10, 9.1000000000000014 },
  { 3360611490639.0889, 10, 9.1999999999999993 },
  { 3768058891466.0469, 10, 9.3000000000000007 },
  { 4219112842239.1055, 10, 9.3999999999999986 },
  { 4717844376391.0000, 10, 9.5000000000000000 },
  { 5268651052510.4785, 10, 9.6000000000000014 },
  { 5876279006180.6377, 10, 9.6999999999999993 },
  { 6545846221520.6768, 10, 9.8000000000000007 },
  { 7282867075495.2949, 10, 9.8999999999999986 },
  { 8093278209760.0000, 10, 10.000000000000000 },
};

// Test function for n=10.
template<typename Tp>
  void
  test005()
  {
    bool test __attribute__((unused)) = true;
    const Tp eps = std::numeric_limits<Tp>::epsilon();
    Tp max_abs_diff = -Tp(1);
    Tp max_abs_frac = -Tp(1);
    unsigned int num_datum = sizeof(data005)
			   / sizeof(testcase_hermite<double>);
    for (unsigned int i = 0; i < num_datum; ++i)
      {
	const Tp f = std::tr1::hermite(Tp(data005[i].n), Tp(data005[i].x));
	const Tp f0 = data005[i].f0;
	const Tp diff = f - f0;
	if (std::abs(diff) > max_abs_diff)
	  max_abs_diff = std::abs(diff);
	if (std::abs(f0) > Tp(10) * eps
	 && std::abs(f) > Tp(10) * eps)
	  {
	    const Tp frac = diff / f0;
	    if (std::abs(frac) > max_abs_frac)
	      max_abs_frac = std::abs(frac);
	  }
      }
    VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
  }

// Test data for n=20.
// max(|f - f_GSL|): 0.0000000000000000
// max(|f - f_GSL| / |f_GSL|): 0.0000000000000000
testcase_hermite<double>
data006[] =
{
  { 3.6536710970888030e+25, 20, -10.000000000000000 },
  { 2.9174932703098834e+25, 20, -9.9000000000000004 },
  { 2.3228769039548404e+25, 20, -9.8000000000000007 },
  { 1.8439144509496016e+25, 20, -9.6999999999999993 },
  { 1.4591971834545420e+25, 20, -9.5999999999999996 },
  { 1.1510764882450827e+25, 20, -9.5000000000000000 },
  { 9.0503941245991605e+24, 20, -9.4000000000000004 },
  { 7.0918188910312152e+24, 20, -9.3000000000000007 },
  { 5.5376531405202033e+24, 20, -9.1999999999999993 },
  { 4.3084410724035914e+24, 20, -9.0999999999999996 },
  { 3.3395360269524137e+24, 20, -9.0000000000000000 },
  { 2.5784910430831484e+24, 20, -8.9000000000000004 },
  { 1.9828824261062853e+24, 20, -8.8000000000000007 },
  { 1.5184989558235974e+24, 20, -8.6999999999999993 },
  { 1.1578391431515818e+24, 20, -8.5999999999999996 },
  { 8.7886740525023878e+23, 20, -8.5000000000000000 },
  { 6.6398733801206072e+23, 20, -8.4000000000000004 },
  { 4.9919656538201190e+23, 20, -8.3000000000000007 },
  { 3.7339306542317994e+23, 20, -8.1999999999999993 },
  { 2.7780752653950559e+23, 20, -8.0999999999999996 },
  { 2.0554027373991249e+23, 20, -8.0000000000000000 },
  { 1.5118471231121695e+23, 20, -7.9000000000000004 },
  { 1.1052214218386250e+23, 20, -7.7999999999999998 },
  { 8.0275268594719504e+22, 20, -7.7000000000000002 },
  { 5.7909832934009042e+22, 20, -7.5999999999999996 },
  { 4.1475563998692745e+22, 20, -7.5000000000000000 },
  { 2.9479149729249250e+22, 20, -7.4000000000000004 },
  { 2.0783192485964573e+22, 20, -7.2999999999999998 },
  { 1.4526171451238503e+22, 20, -7.2000000000000002 },
  { 1.0059323685871305e+22, 20, -7.0999999999999996 },
  { 6.8970965604502329e+21, 20, -7.0000000000000000 },
  { 4.6784410379528280e+21, 20, -6.9000000000000004 },
  { 3.1367268160419670e+21, 20, -6.7999999999999998 },
  { 2.0764831558419748e+21, 20, -6.7000000000000002 },
  { 1.3555174744148132e+21, 20, -6.5999999999999996 },
  { 8.7124954970091579e+20, 20, -6.5000000000000000 },
  { 5.5033278133788108e+20, 20, -6.4000000000000004 },
  { 3.4082412197533739e+20, 20, -6.2999999999999998 },
  { 2.0631845648712185e+20, 20, -6.2000000000000002 },
  { 1.2158762212269028e+20, 20, -6.0999999999999996 },
  { 6.9364200641629315e+19, 20, -6.0000000000000000 },
  { 3.7990093270095905e+19, 20, -5.9000000000000004 },
  { 1.9713403927925858e+19, 20, -5.7999999999999998 },
  { 9.4673798488533340e+18, 20, -5.7000000000000002 },
  { 4.0046403628787825e+18, 20, -5.5999999999999996 },
  { 1.2907686705724293e+18, 20, -5.5000000000000000 },
  { 85277679782823936., 20, -5.4000000000000004 },
  { -3.4359547179069376e+17, 20, -5.2999999999999998 },
  { -4.0970873501577555e+17, 20, -5.2000000000000002 },
  { -3.3419585613348250e+17, 20, -5.0999999999999996 },
  { -2.2571776716382720e+17, 20, -5.0000000000000000 },
  { -1.3051120203565566e+17, 20, -4.9000000000000004 },
  { -62555669197021992., 20, -4.7999999999999998 },
  { -20974173561010048., 20, -4.7000000000000002 },
  { 519073301695656.00, 20, -4.5999999999999996 },
  { 8939556693761121.0, 20, -4.5000000000000000 },
  { 10070625675125180., 20, -4.4000000000000004 },
  { 7940371996960715.0, 20, -4.2999999999999998 },
  { 4973623686173568.0, 20, -4.2000000000000002 },
  { 2389023282480410.0, 20, -4.0999999999999996 },
  { 619678016654336.00, 20, -4.0000000000000000 },
  { -339773491011946.00, 20, -3.9000000000000004 },
  { -687467334428204.38, 20, -3.7999999999999998 },
  { -663019129550482.75, 20, -3.7000000000000002 },
  { -469585152350670.25, 20, -3.5999999999999996 },
  { -245659781875039.00, 20, -3.5000000000000000 },
  { -66042773886778.938, 20, -3.4000000000000004 },
  { 43442261337436.312, 20, -3.2999999999999998 },
  { 87626632986465.375, 20, -3.2000000000000002 },
  { 85786202388511.375, 20, -3.0999999999999996 },
  { 59990281399296.000, 20, -3.0000000000000000 },
  { 28343496696718.070, 20, -2.9000000000000004 },
  { 2296898915036.0859, 20, -2.7999999999999998 },
  { -13249381789941.502, 20, -2.7000000000000002 },
  { -18328180426561.059, 20, -2.5999999999999996 },
  { -15799429886575.000, 20, -2.5000000000000000 },
  { -9442592050214.3027, 20, -2.4000000000000004 },
  { -2602375356373.2393, 20, -2.2999999999999998 },
  { 2521759315047.8428, 20, -2.2000000000000002 },
  { 5027779307352.6660, 20, -2.0999999999999996 },
  { 5080118660096.0000, 20, -2.0000000000000000 },
  { 3490463276000.5425, 20, -1.9000000000000004 },
  { 1277254793997.1128, 20, -1.8000000000000007 },
  { -682119355279.28784, 20, -1.6999999999999993 },
  { -1851544254412.3203, 20, -1.5999999999999996 },
  { -2085387081039.0000, 20, -1.5000000000000000 },
  { -1559402933581.5054, 20, -1.4000000000000004 },
  { -634265763221.87231, 20, -1.3000000000000007 },
  { 295481874887.33429, 20, -1.1999999999999993 },
  { 924603483429.34241, 20, -1.0999999999999996 },
  { 1107214478336.0000, 20, -1.0000000000000000 },
  { 867235623835.12463, 20, -0.90000000000000036 },
  { 358848462745.15466, 20, -0.80000000000000071 },
  { -202944086511.71936, 20, -0.69999999999999929 },
  { -617730863561.32617, 20, -0.59999999999999964 },
  { -759627879679.00000, 20, -0.50000000000000000 },
  { -607451728035.03271, 20, -0.40000000000000036 },
  { -240424854484.42111, 20, -0.30000000000000071 },
  { 195759209122.61337, 20, -0.19999999999999929 },
  { 540334019322.52014, 20, -0.099999999999999645 },
  { 670442572800.00000, 20, 0.0000000000000000 },
  { 540334019322.52014, 20, 0.099999999999999645 },
  { 195759209122.61337, 20, 0.19999999999999929 },
  { -240424854484.42111, 20, 0.30000000000000071 },
  { -607451728035.03271, 20, 0.40000000000000036 },
  { -759627879679.00000, 20, 0.50000000000000000 },
  { -617730863561.32617, 20, 0.59999999999999964 },
  { -202944086511.71936, 20, 0.69999999999999929 },
  { 358848462745.15466, 20, 0.80000000000000071 },
  { 867235623835.12463, 20, 0.90000000000000036 },
  { 1107214478336.0000, 20, 1.0000000000000000 },
  { 924603483429.34241, 20, 1.0999999999999996 },
  { 295481874887.33429, 20, 1.1999999999999993 },
  { -634265763221.87231, 20, 1.3000000000000007 },
  { -1559402933581.5054, 20, 1.4000000000000004 },
  { -2085387081039.0000, 20, 1.5000000000000000 },
  { -1851544254412.3203, 20, 1.5999999999999996 },
  { -682119355279.28784, 20, 1.6999999999999993 },
  { 1277254793997.1128, 20, 1.8000000000000007 },
  { 3490463276000.5425, 20, 1.9000000000000004 },
  { 5080118660096.0000, 20, 2.0000000000000000 },
  { 5027779307352.6660, 20, 2.0999999999999996 },
  { 2521759315047.8770, 20, 2.1999999999999993 },
  { -2602375356373.2969, 20, 2.3000000000000007 },
  { -9442592050214.3027, 20, 2.4000000000000004 },
  { -15799429886575.000, 20, 2.5000000000000000 },
  { -18328180426561.059, 20, 2.5999999999999996 },
  { -13249381789941.586, 20, 2.6999999999999993 },
  { 2296898915036.2812, 20, 2.8000000000000007 },
  { 28343496696718.070, 20, 2.9000000000000004 },
  { 59990281399296.000, 20, 3.0000000000000000 },
  { 85786202388511.375, 20, 3.0999999999999996 },
  { 87626632986465.438, 20, 3.1999999999999993 },
  { 43442261337435.672, 20, 3.3000000000000007 },
  { -66042773886778.938, 20, 3.4000000000000004 },
  { -245659781875039.00, 20, 3.5000000000000000 },
  { -469585152350670.25, 20, 3.5999999999999996 },
  { -663019129550482.25, 20, 3.6999999999999993 },
  { -687467334428203.38, 20, 3.8000000000000007 },
  { -339773491011946.00, 20, 3.9000000000000004 },
  { 619678016654336.00, 20, 4.0000000000000000 },
  { 2389023282480410.0, 20, 4.0999999999999996 },
  { 4973623686173539.0, 20, 4.1999999999999993 },
  { 7940371996960741.0, 20, 4.3000000000000007 },
  { 10070625675125180., 20, 4.4000000000000004 },
  { 8939556693761121.0, 20, 4.5000000000000000 },
  { 519073301695656.00, 20, 4.5999999999999996 },
  { -20974173561009776., 20, 4.6999999999999993 },
  { -62555669197022528., 20, 4.8000000000000007 },
  { -1.3051120203565566e+17, 20, 4.9000000000000004 },
  { -2.2571776716382720e+17, 20, 5.0000000000000000 },
  { -3.3419585613348250e+17, 20, 5.0999999999999996 },
  { -4.0970873501577562e+17, 20, 5.1999999999999993 },
  { -3.4359547179069216e+17, 20, 5.3000000000000007 },
  { 85277679782823936., 20, 5.4000000000000004 },
  { 1.2907686705724293e+18, 20, 5.5000000000000000 },
  { 4.0046403628787825e+18, 20, 5.5999999999999996 },
  { 9.4673798488532767e+18, 20, 5.6999999999999993 },
  { 1.9713403927925973e+19, 20, 5.8000000000000007 },
  { 3.7990093270095905e+19, 20, 5.9000000000000004 },
  { 6.9364200641629315e+19, 20, 6.0000000000000000 },
  { 1.2158762212269156e+20, 20, 6.1000000000000014 },
  { 2.0631845648712086e+20, 20, 6.1999999999999993 },
  { 3.4082412197533902e+20, 20, 6.3000000000000007 },
  { 5.5033278133787696e+20, 20, 6.3999999999999986 },
  { 8.7124954970091579e+20, 20, 6.5000000000000000 },
  { 1.3555174744148243e+21, 20, 6.6000000000000014 },
  { 2.0764831558419680e+21, 20, 6.6999999999999993 },
  { 3.1367268160419775e+21, 20, 6.8000000000000007 },
  { 4.6784410379527966e+21, 20, 6.8999999999999986 },
  { 6.8970965604502329e+21, 20, 7.0000000000000000 },
  { 1.0059323685871368e+22, 20, 7.1000000000000014 },
  { 1.4526171451238465e+22, 20, 7.1999999999999993 },
  { 2.0783192485964666e+22, 20, 7.3000000000000007 },
  { 2.9479149729249048e+22, 20, 7.3999999999999986 },
  { 4.1475563998692745e+22, 20, 7.5000000000000000 },
  { 5.7909832934009378e+22, 20, 7.6000000000000014 },
  { 8.0275268594719286e+22, 20, 7.6999999999999993 },
  { 1.1052214218386286e+23, 20, 7.8000000000000007 },
  { 1.5118471231121604e+23, 20, 7.8999999999999986 },
  { 2.0554027373991249e+23, 20, 8.0000000000000000 },
  { 2.7780752653950703e+23, 20, 8.1000000000000014 },
  { 3.7339306542317994e+23, 20, 8.1999999999999993 },
  { 4.9919656538201190e+23, 20, 8.3000000000000007 },
  { 6.6398733801205790e+23, 20, 8.3999999999999986 },
  { 8.7886740525023878e+23, 20, 8.5000000000000000 },
  { 1.1578391431515869e+24, 20, 8.6000000000000014 },
  { 1.5184989558235974e+24, 20, 8.6999999999999993 },
  { 1.9828824261062853e+24, 20, 8.8000000000000007 },
  { 2.5784910430831355e+24, 20, 8.8999999999999986 },
  { 3.3395360269524137e+24, 20, 9.0000000000000000 },
  { 4.3084410724036123e+24, 20, 9.1000000000000014 },
  { 5.5376531405202033e+24, 20, 9.1999999999999993 },
  { 7.0918188910312152e+24, 20, 9.3000000000000007 },
  { 9.0503941245991197e+24, 20, 9.3999999999999986 },
  { 1.1510764882450827e+25, 20, 9.5000000000000000 },
  { 1.4591971834545491e+25, 20, 9.6000000000000014 },
  { 1.8439144509496016e+25, 20, 9.6999999999999993 },
  { 2.3228769039548404e+25, 20, 9.8000000000000007 },
  { 2.9174932703098731e+25, 20, 9.8999999999999986 },
  { 3.6536710970888030e+25, 20, 10.000000000000000 },
};

// Test function for n=20.
template<typename Tp>
  void
  test006()
  {
    bool test __attribute__((unused)) = true;
    const Tp eps = std::numeric_limits<Tp>::epsilon();
    Tp max_abs_diff = -Tp(1);
    Tp max_abs_frac = -Tp(1);
    unsigned int num_datum = sizeof(data006)
			   / sizeof(testcase_hermite<double>);
    for (unsigned int i = 0; i < num_datum; ++i)
      {
	const Tp f = std::tr1::hermite(Tp(data006[i].n), Tp(data006[i].x));
	const Tp f0 = data006[i].f0;
	const Tp diff = f - f0;
	if (std::abs(diff) > max_abs_diff)
	  max_abs_diff = std::abs(diff);
	if (std::abs(f0) > Tp(10) * eps
	 && std::abs(f) > Tp(10) * eps)
	  {
	    const Tp frac = diff / f0;
	    if (std::abs(frac) > max_abs_frac)
	      max_abs_frac = std::abs(frac);
	  }
      }
    VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
  }

// Test data for n=50.
// max(|f - f_GSL|): 0.0000000000000000
// max(|f - f_GSL| / |f_GSL|): 0.0000000000000000
testcase_hermite<double>
data007[] =
{
  { 1.3516643049819314e+61, 50, -10.000000000000000 },
  { 5.8466703062266110e+60, 50, -9.9000000000000004 },
  { 2.4344814863741168e+60, 50, -9.8000000000000007 },
  { 9.6739828066585787e+59, 50, -9.6999999999999993 },
  { 3.6194312967780128e+59, 50, -9.5999999999999996 },
  { 1.2454022521696361e+59, 50, -9.5000000000000000 },
  { 3.7546881918013145e+58, 50, -9.4000000000000004 },
  { 8.6500158470917270e+57, 50, -9.3000000000000007 },
  { 5.3163209769527426e+56, 50, -9.1999999999999993 },
  { -9.7863165370930473e+56, 50, -9.0999999999999996 },
  { -8.0563930902368911e+56, 50, -9.0000000000000000 },
  { -4.3145937406321933e+56, 50, -8.9000000000000004 },
  { -1.8210771577645630e+56, 50, -8.8000000000000007 },
  { -6.0082225302289557e+55, 50, -8.6999999999999993 },
  { -1.2392013720948442e+55, 50, -8.5999999999999996 },
  { 1.5935887905806307e+54, 50, -8.5000000000000000 },
  { 3.4325365049909381e+54, 50, -8.4000000000000004 },
  { 2.2368641272300899e+54, 50, -8.3000000000000007 },
  { 1.0009712198862341e+54, 50, -8.1999999999999993 },
  { 3.0699480272357337e+53, 50, -8.0999999999999996 },
  { 2.9492797132685063e+52, 50, -8.0000000000000000 },
  { -3.9982544106300062e+52, 50, -7.9000000000000004 },
  { -3.5678031330023779e+52, 50, -7.7999999999999998 },
  { -1.8076371748762468e+52, 50, -7.7000000000000002 },
  { -5.7887622198708925e+51, 50, -7.5999999999999996 },
  { -3.5808467693113330e+50, 50, -7.5000000000000000 },
  { 1.0219511166806405e+51, 50, -7.4000000000000004 },
  { 8.4241263694549560e+50, 50, -7.2999999999999998 },
  { 3.9143899315095369e+50, 50, -7.2000000000000002 },
  { 9.3178966245756654e+49, 50, -7.0999999999999996 },
  { -2.4714855983436561e+49, 50, -7.0000000000000000 },
  { -4.1428217272044600e+49, 50, -6.9000000000000004 },
  { -2.4864980414057334e+49, 50, -6.7999999999999998 },
  { -8.0684355476447979e+48, 50, -6.7000000000000002 },
  { 4.2529238530179841e+47, 50, -6.5999999999999996 },
  { 2.4709663739849681e+48, 50, -6.5000000000000000 },
  { 1.7500305481204125e+48, 50, -6.4000000000000004 },
  { 6.3834558285146007e+47, 50, -6.2999999999999998 },
  { -1.1477672402381055e+46, 50, -6.2000000000000002 },
  { -1.9304630401841983e+47, 50, -6.0999999999999996 },
  { -1.4355266959903589e+47, 50, -6.0000000000000000 },
  { -5.1482200905566146e+46, 50, -5.9000000000000004 },
  { 4.6577356827463283e+45, 50, -5.7999999999999998 },
  { 1.9676012349652066e+46, 50, -5.7000000000000002 },
  { 1.3630554018675846e+46, 50, -5.5999999999999996 },
  { 4.0920495328093750e+45, 50, -5.5000000000000000 },
  { -1.3680638882145392e+45, 50, -5.4000000000000004 },
  { -2.4465265559935436e+45, 50, -5.2999999999999998 },
  { -1.4270495629649456e+45, 50, -5.2000000000000002 },
  { -2.4845518743338381e+44, 50, -5.0999999999999996 },
  { 3.1953926721271990e+44, 50, -5.0000000000000000 },
  { 3.4169399444879600e+44, 50, -4.9000000000000004 },
  { 1.4896819114978755e+44, 50, -4.7999999999999998 },
  { -1.3078430866968493e+43, 50, -4.7000000000000002 },
  { -6.8449168639700716e+43, 50, -4.5999999999999996 },
  { -4.9181639709997461e+43, 50, -4.5000000000000000 },
  { -1.1434943490345182e+43, 50, -4.4000000000000004 },
  { 1.1214652543461432e+43, 50, -4.2999999999999998 },
  { 1.3843548994480608e+43, 50, -4.2000000000000002 },
  { 6.3349790205303262e+42, 50, -4.0999999999999996 },
  { -9.5599640670553907e+41, 50, -4.0000000000000000 },
  { -3.6202558158287927e+42, 50, -3.9000000000000004 },
  { -2.5206461734624493e+42, 50, -3.7999999999999998 },
  { -3.7818051510732439e+41, 50, -3.7000000000000002 },
  { 8.8921371165335050e+41, 50, -3.5999999999999996 },
  { 9.2055485763852770e+41, 50, -3.5000000000000000 },
  { 3.2535765707283020e+41, 50, -3.4000000000000004 },
  { -1.9358941418244578e+41, 50, -3.2999999999999998 },
  { -3.3076282847915670e+41, 50, -3.2000000000000002 },
  { -1.7764977066639160e+41, 50, -3.0999999999999996 },
  { 2.6751882008697154e+40, 50, -3.0000000000000000 },
  { 1.2025382369996052e+41, 50, -2.9000000000000004 },
  { 8.8383035103557973e+40, 50, -2.7999999999999998 },
  { 7.7733606479641769e+39, 50, -2.7000000000000002 },
  { -4.4696811758713757e+40, 50, -2.5999999999999996 },
  { -4.3715062488963453e+40, 50, -2.5000000000000000 },
  { -1.1390080390575289e+40, 50, -2.4000000000000004 },
  { 1.6938519751181342e+40, 50, -2.2999999999999998 },
  { 2.2284509952956210e+40, 50, -2.2000000000000002 },
  { 9.0967994280570531e+39, 50, -2.0999999999999996 },
  { -6.4126677997472978e+39, 50, -2.0000000000000000 },
  { -1.1926839454034341e+40, 50, -1.9000000000000004 },
  { -6.5436654274699114e+39, 50, -1.8000000000000007 },
  { 2.2779499542550411e+39, 50, -1.6999999999999993 },
  { 6.7720530889699639e+39, 50, -1.5999999999999996 },
  { 4.6884851188034300e+39, 50, -1.5000000000000000 },
  { -5.9005001052557463e+38, 50, -1.4000000000000004 },
  { -4.1028320210430589e+39, 50, -1.3000000000000007 },
  { -3.4780049977083965e+39, 50, -1.1999999999999993 },
  { -1.3484918476373692e+38, 50, -1.0999999999999996 },
  { 2.6586815431645456e+39, 50, -1.0000000000000000 },
  { 2.7225429473661429e+39, 50, -0.90000000000000036 },
  { 4.7785441024951729e+38, 50, -0.80000000000000071 },
  { -1.8416784378790159e+39, 50, -0.69999999999999929 },
  { -2.2725918816693132e+39, 50, -0.59999999999999964 },
  { -6.7948375014926916e+38, 50, -0.50000000000000000 },
  { 1.3581645858905750e+39, 50, -0.40000000000000036 },
  { 2.0349661043040497e+39, 50, -0.30000000000000071 },
  { 8.5049248815817037e+38, 50, -0.19999999999999929 },
  { -1.0564542132990048e+39, 50, -0.099999999999999645 },
  { -1.9607814681608194e+39, 50, 0.0000000000000000 },
  { -1.0564542132990048e+39, 50, 0.099999999999999645 },
  { 8.5049248815817037e+38, 50, 0.19999999999999929 },
  { 2.0349661043040497e+39, 50, 0.30000000000000071 },
  { 1.3581645858905750e+39, 50, 0.40000000000000036 },
  { -6.7948375014926916e+38, 50, 0.50000000000000000 },
  { -2.2725918816693132e+39, 50, 0.59999999999999964 },
  { -1.8416784378790159e+39, 50, 0.69999999999999929 },
  { 4.7785441024951729e+38, 50, 0.80000000000000071 },
  { 2.7225429473661429e+39, 50, 0.90000000000000036 },
  { 2.6586815431645456e+39, 50, 1.0000000000000000 },
  { -1.3484918476373692e+38, 50, 1.0999999999999996 },
  { -3.4780049977083965e+39, 50, 1.1999999999999993 },
  { -4.1028320210430589e+39, 50, 1.3000000000000007 },
  { -5.9005001052557463e+38, 50, 1.4000000000000004 },
  { 4.6884851188034300e+39, 50, 1.5000000000000000 },
  { 6.7720530889699639e+39, 50, 1.5999999999999996 },
  { 2.2779499542550411e+39, 50, 1.6999999999999993 },
  { -6.5436654274699114e+39, 50, 1.8000000000000007 },
  { -1.1926839454034341e+40, 50, 1.9000000000000004 },
  { -6.4126677997472978e+39, 50, 2.0000000000000000 },
  { 9.0967994280570531e+39, 50, 2.0999999999999996 },
  { 2.2284509952956162e+40, 50, 2.1999999999999993 },
  { 1.6938519751181172e+40, 50, 2.3000000000000007 },
  { -1.1390080390575289e+40, 50, 2.4000000000000004 },
  { -4.3715062488963453e+40, 50, 2.5000000000000000 },
  { -4.4696811758713757e+40, 50, 2.5999999999999996 },
  { 7.7733606479635628e+39, 50, 2.6999999999999993 },
  { 8.8383035103558611e+40, 50, 2.8000000000000007 },
  { 1.2025382369996052e+41, 50, 2.9000000000000004 },
  { 2.6751882008697154e+40, 50, 3.0000000000000000 },
  { -1.7764977066639160e+41, 50, 3.0999999999999996 },
  { -3.3076282847915616e+41, 50, 3.1999999999999993 },
  { -1.9358941418244260e+41, 50, 3.3000000000000007 },
  { 3.2535765707283020e+41, 50, 3.4000000000000004 },
  { 9.2055485763852770e+41, 50, 3.5000000000000000 },
  { 8.8921371165335050e+41, 50, 3.5999999999999996 },
  { -3.7818051510730675e+41, 50, 3.6999999999999993 },
  { -2.5206461734624660e+42, 50, 3.8000000000000007 },
  { -3.6202558158287927e+42, 50, 3.9000000000000004 },
  { -9.5599640670553907e+41, 50, 4.0000000000000000 },
  { 6.3349790205303262e+42, 50, 4.0999999999999996 },
  { 1.3843548994480566e+43, 50, 4.1999999999999993 },
  { 1.1214652543461340e+43, 50, 4.3000000000000007 },
  { -1.1434943490345182e+43, 50, 4.4000000000000004 },
  { -4.9181639709997461e+43, 50, 4.5000000000000000 },
  { -6.8449168639700716e+43, 50, 4.5999999999999996 },
  { -1.3078430866969463e+43, 50, 4.6999999999999993 },
  { 1.4896819114978953e+44, 50, 4.8000000000000007 },
  { 3.4169399444879600e+44, 50, 4.9000000000000004 },
  { 3.1953926721271990e+44, 50, 5.0000000000000000 },
  { -2.4845518743338381e+44, 50, 5.0999999999999996 },
  { -1.4270495629649337e+45, 50, 5.1999999999999993 },
  { -2.4465265559935458e+45, 50, 5.3000000000000007 },
  { -1.3680638882145392e+45, 50, 5.4000000000000004 },
  { 4.0920495328093750e+45, 50, 5.5000000000000000 },
  { 1.3630554018675846e+46, 50, 5.5999999999999996 },
  { 1.9676012349652081e+46, 50, 5.6999999999999993 },
  { 4.6577356827460393e+45, 50, 5.8000000000000007 },
  { -5.1482200905566146e+46, 50, 5.9000000000000004 },
  { -1.4355266959903589e+47, 50, 6.0000000000000000 },
  { -1.9304630401841966e+47, 50, 6.1000000000000014 },
  { -1.1477672402384868e+46, 50, 6.1999999999999993 },
  { 6.3834558285146981e+47, 50, 6.3000000000000007 },
  { 1.7500305481203924e+48, 50, 6.3999999999999986 },
  { 2.4709663739849681e+48, 50, 6.5000000000000000 },
  { 4.2529238530171793e+47, 50, 6.6000000000000014 },
  { -8.0684355476446876e+48, 50, 6.6999999999999993 },
  { -2.4864980414057495e+49, 50, 6.8000000000000007 },
  { -4.1428217272044496e+49, 50, 6.8999999999999986 },
  { -2.4714855983436561e+49, 50, 7.0000000000000000 },
  { 9.3178966245760310e+49, 50, 7.1000000000000014 },
  { 3.9143899315095070e+50, 50, 7.1999999999999993 },
  { 8.4241263694549925e+50, 50, 7.3000000000000007 },
  { 1.0219511166806458e+51, 50, 7.3999999999999986 },
  { -3.5808467693113330e+50, 50, 7.5000000000000000 },
  { -5.7887622198710268e+51, 50, 7.6000000000000014 },
  { -1.8076371748762319e+52, 50, 7.6999999999999993 },
  { -3.5678031330023971e+52, 50, 7.8000000000000007 },
  { -3.9982544106300530e+52, 50, 7.8999999999999986 },
  { 2.9492797132685063e+52, 50, 8.0000000000000000 },
  { 3.0699480272358086e+53, 50, 8.1000000000000014 },
  { 1.0009712198862341e+54, 50, 8.1999999999999993 },
  { 2.2368641272300899e+54, 50, 8.3000000000000007 },
  { 3.4325365049909340e+54, 50, 8.3999999999999986 },
  { 1.5935887905806307e+54, 50, 8.5000000000000000 },
  { -1.2392013720948937e+55, 50, 8.6000000000000014 },
  { -6.0082225302289557e+55, 50, 8.6999999999999993 },
  { -1.8210771577645630e+56, 50, 8.8000000000000007 },
  { -4.3145937406321376e+56, 50, 8.8999999999999986 },
  { -8.0563930902368911e+56, 50, 9.0000000000000000 },
  { -9.7863165370930194e+56, 50, 9.1000000000000014 },
  { 5.3163209769527426e+56, 50, 9.1999999999999993 },
  { 8.6500158470917270e+57, 50, 9.3000000000000007 },
  { 3.7546881918012164e+58, 50, 9.3999999999999986 },
  { 1.2454022521696361e+59, 50, 9.5000000000000000 },
  { 3.6194312967780793e+59, 50, 9.6000000000000014 },
  { 9.6739828066585787e+59, 50, 9.6999999999999993 },
  { 2.4344814863741168e+60, 50, 9.8000000000000007 },
  { 5.8466703062265247e+60, 50, 9.8999999999999986 },
  { 1.3516643049819314e+61, 50, 10.000000000000000 },
};

// Test function for n=50.
template<typename Tp>
  void
  test007()
  {
    bool test __attribute__((unused)) = true;
    const Tp eps = std::numeric_limits<Tp>::epsilon();
    Tp max_abs_diff = -Tp(1);
    Tp max_abs_frac = -Tp(1);
    unsigned int num_datum = sizeof(data007)
			   / sizeof(testcase_hermite<double>);
    for (unsigned int i = 0; i < num_datum; ++i)
      {
	const Tp f = std::tr1::hermite(Tp(data007[i].n), Tp(data007[i].x));
	const Tp f0 = data007[i].f0;
	const Tp diff = f - f0;
	if (std::abs(diff) > max_abs_diff)
	  max_abs_diff = std::abs(diff);
	if (std::abs(f0) > Tp(10) * eps
	 && std::abs(f) > Tp(10) * eps)
	  {
	    const Tp frac = diff / f0;
	    if (std::abs(frac) > max_abs_frac)
	      max_abs_frac = std::abs(frac);
	  }
      }
    VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
  }

// Test data for n=100.
// max(|f - f_GSL|): 0.0000000000000000
// max(|f - f_GSL| / |f_GSL|): 0.0000000000000000
testcase_hermite<double>
data008[] =
{
  { -1.8738689223256612e+115, 100, -10.000000000000000 },
  { -4.4232904120311186e+114, 100, -9.9000000000000004 },
  { 8.3761676654305186e+113, 100, -9.8000000000000007 },
  { 9.4857435427570856e+113, 100, -9.6999999999999993 },
  { 2.4904626130429828e+113, 100, -9.5999999999999996 },
  { -4.2265051766934789e+112, 100, -9.5000000000000000 },
  { -5.3374224541142079e+112, 100, -9.4000000000000004 },
  { -1.3977488190034162e+112, 100, -9.3000000000000007 },
  { 2.9686037583393142e+111, 100, -9.1999999999999993 },
  { 3.3341330166544008e+111, 100, -9.0999999999999996 },
  { 7.7294136683043515e+110, 100, -9.0000000000000000 },
  { -2.6071876743328352e+110, 100, -8.9000000000000004 },
  { -2.2669773300293168e+110, 100, -8.8000000000000007 },
  { -3.8930624477368766e+109, 100, -8.6999999999999993 },
  { 2.5386102824684956e+109, 100, -8.5999999999999996 },
  { 1.6203702280499544e+109, 100, -8.5000000000000000 },
  { 1.2481735250539652e+108, 100, -8.4000000000000004 },
  { -2.5611278359589723e+108, 100, -8.3000000000000007 },
  { -1.1534178973583771e+108, 100, -8.1999999999999993 },
  { 8.4932594446954126e+106, 100, -8.0999999999999996 },
  { 2.5761505535443451e+107, 100, -8.0000000000000000 },
  { 7.3445622927437730e+106, 100, -7.9000000000000004 },
  { -2.7252309851017323e+106, 100, -7.7999999999999998 },
  { -2.4850432648124068e+106, 100, -7.7000000000000002 },
  { -2.7486638813238851e+105, 100, -7.5999999999999996 },
  { 4.4772364475394960e+105, 100, -7.5000000000000000 },
  { 2.1375121759800508e+105, 100, -7.4000000000000004 },
  { -2.9237691057265876e+104, 100, -7.2999999999999998 },
  { -5.9348804074874781e+104, 100, -7.2000000000000002 },
  { -1.2947583568692734e+104, 100, -7.0999999999999996 },
  { 1.0002273523275075e+104, 100, -7.0000000000000000 },
  { 6.4900499886057735e+103, 100, -6.9000000000000004 },
  { -3.7817730019124298e+102, 100, -6.7999999999999998 },
  { -1.7849709684883137e+103, 100, -6.7000000000000002 },
  { -4.8039047085580619e+102, 100, -6.5999999999999996 },
  { 3.2072472002047670e+102, 100, -6.5000000000000000 },
  { 2.3341289432999226e+102, 100, -6.4000000000000004 },
  { -1.4587350659146165e+101, 100, -6.2999999999999998 },
  { -7.0672139150705532e+101, 100, -6.2000000000000002 },
  { -1.8039133351601998e+101, 100, -6.0999999999999996 },
  { 1.5170693933260738e+101, 100, -6.0000000000000000 },
  { 1.0141407690966954e+101, 100, -5.9000000000000004 },
  { -1.5140717605507886e+100, 100, -5.7999999999999998 },
  { -3.6379102593330533e+100, 100, -5.7000000000000002 },
  { -6.2933057672779134e+99, 100, -5.5999999999999996 },
  { 9.9309186425970402e+99, 100, -5.5000000000000000 },
  { 5.0935670879056567e+99, 100, -5.4000000000000004 },
  { -1.8337906983006129e+99, 100, -5.2999999999999998 },
  { -2.3096466887887402e+99, 100, -5.2000000000000002 },
  { -4.0259235416602546e+97, 100, -5.0999999999999996 },
  { 8.1931937130591466e+98, 100, -5.0000000000000000 },
  { 2.5601172475776894e+98, 100, -4.9000000000000004 },
  { -2.3193021196802698e+98, 100, -4.7999999999999998 },
  { -1.6545238014142802e+98, 100, -4.7000000000000002 },
  { 4.2440218943299170e+97, 100, -4.5999999999999996 },
  { 7.8073853756187101e+97, 100, -4.5000000000000000 },
  { 5.2136163919277588e+96, 100, -4.4000000000000004 },
  { -3.0837503404959950e+97, 100, -4.2999999999999998 },
  { -1.0988290456447777e+97, 100, -4.2000000000000002 },
  { 1.0173847230741884e+97, 100, -4.0999999999999996 },
  { 7.7070682747643550e+96, 100, -4.0000000000000000 },
  { -2.3947326745293331e+96, 100, -3.9000000000000004 },
  { -4.1993887070961596e+96, 100, -3.7999999999999998 },
  { -5.5521768050542138e+94, 100, -3.7000000000000002 },
  { 1.9852724296241180e+96, 100, -3.5999999999999996 },
  { 5.7841213444838915e+95, 100, -3.5000000000000000 },
  { -8.2433116644948570e+95, 100, -3.4000000000000004 },
  { -5.2035951331058195e+95, 100, -3.2999999999999998 },
  { 2.8212012454566057e+95, 100, -3.2000000000000002 },
  { 3.5161870090563867e+95, 100, -3.0999999999999996 },
  { -5.5084542871196523e+94, 100, -3.0000000000000000 },
  { -2.0784160746797610e+95, 100, -2.9000000000000004 },
  { -2.6058116248679496e+94, 100, -2.7999999999999998 },
  { 1.1147132226581844e+95, 100, -2.7000000000000002 },
  { 4.5607706742532875e+94, 100, -2.5999999999999996 },
  { -5.3758761713337664e+94, 100, -2.5000000000000000 },
  { -4.2303228932575769e+94, 100, -2.4000000000000004 },
  { 2.1691501564685499e+94, 100, -2.2999999999999998 },
  { 3.2602488340116974e+94, 100, -2.2000000000000002 },
  { -5.0527155039787607e+93, 100, -2.0999999999999996 },
  { -2.2785574311661325e+94, 100, -2.0000000000000000 },
  { -2.8549527653152903e+93, 100, -1.9000000000000004 },
  { 1.4787573463714363e+94, 100, -1.8000000000000007 },
  { 6.0554070654771248e+93, 100, -1.6999999999999993 },
  { -8.8496828346970978e+93, 100, -1.5999999999999996 },
  { -6.8402151897169509e+93, 100, -1.5000000000000000 },
  { 4.6555468819923166e+93, 100, -1.4000000000000004 },
  { 6.4625437128323579e+93, 100, -1.3000000000000007 },
  { -1.7820042440391653e+93, 100, -1.1999999999999993 },
  { -5.5814393347235886e+93, 100, -1.0999999999999996 },
  { -1.4487067293379347e+92, 100, -1.0000000000000000 },
  { 4.5268398678911204e+93, 100, -0.90000000000000036 },
  { 1.4120762149478435e+93, 100, -0.80000000000000071 },
  { -3.4510765981144258e+93, 100, -0.69999999999999929 },
  { -2.2242581581553176e+93, 100, -0.59999999999999964 },
  { 2.4129827902061037e+93, 100, -0.50000000000000000 },
  { 2.7195429139752497e+93, 100, -0.40000000000000036 },
  { -1.4235309630836904e+93, 100, -0.30000000000000071 },
  { -2.9850618739468043e+93, 100, -0.19999999999999929 },
  { 4.7017027479251074e+92, 100, -0.099999999999999645 },
  { 3.0685187562549660e+93, 100, 0.0000000000000000 },
  { 4.7017027479251074e+92, 100, 0.099999999999999645 },
  { -2.9850618739468043e+93, 100, 0.19999999999999929 },
  { -1.4235309630836904e+93, 100, 0.30000000000000071 },
  { 2.7195429139752497e+93, 100, 0.40000000000000036 },
  { 2.4129827902061037e+93, 100, 0.50000000000000000 },
  { -2.2242581581553176e+93, 100, 0.59999999999999964 },
  { -3.4510765981144258e+93, 100, 0.69999999999999929 },
  { 1.4120762149478435e+93, 100, 0.80000000000000071 },
  { 4.5268398678911204e+93, 100, 0.90000000000000036 },
  { -1.4487067293379347e+92, 100, 1.0000000000000000 },
  { -5.5814393347235886e+93, 100, 1.0999999999999996 },
  { -1.7820042440391653e+93, 100, 1.1999999999999993 },
  { 6.4625437128323579e+93, 100, 1.3000000000000007 },
  { 4.6555468819923166e+93, 100, 1.4000000000000004 },
  { -6.8402151897169509e+93, 100, 1.5000000000000000 },
  { -8.8496828346970978e+93, 100, 1.5999999999999996 },
  { 6.0554070654771248e+93, 100, 1.6999999999999993 },
  { 1.4787573463714363e+94, 100, 1.8000000000000007 },
  { -2.8549527653152903e+93, 100, 1.9000000000000004 },
  { -2.2785574311661325e+94, 100, 2.0000000000000000 },
  { -5.0527155039787607e+93, 100, 2.0999999999999996 },
  { 3.2602488340116774e+94, 100, 2.1999999999999993 },
  { 2.1691501564685076e+94, 100, 2.3000000000000007 },
  { -4.2303228932575769e+94, 100, 2.4000000000000004 },
  { -5.3758761713337664e+94, 100, 2.5000000000000000 },
  { 4.5607706742532875e+94, 100, 2.5999999999999996 },
  { 1.1147132226581881e+95, 100, 2.6999999999999993 },
  { -2.6058116248681564e+94, 100, 2.8000000000000007 },
  { -2.0784160746797610e+95, 100, 2.9000000000000004 },
  { -5.5084542871196523e+94, 100, 3.0000000000000000 },
  { 3.5161870090563867e+95, 100, 3.0999999999999996 },
  { 2.8212012454566478e+95, 100, 3.1999999999999993 },
  { -5.2035951331058918e+95, 100, 3.3000000000000007 },
  { -8.2433116644948570e+95, 100, 3.4000000000000004 },
  { 5.7841213444838915e+95, 100, 3.5000000000000000 },
  { 1.9852724296241180e+96, 100, 3.5999999999999996 },
  { -5.5521768050503009e+94, 100, 3.6999999999999993 },
  { -4.1993887070961795e+96, 100, 3.8000000000000007 },
  { -2.3947326745293331e+96, 100, 3.9000000000000004 },
  { 7.7070682747643550e+96, 100, 4.0000000000000000 },
  { 1.0173847230741884e+97, 100, 4.0999999999999996 },
  { -1.0988290456447506e+97, 100, 4.1999999999999993 },
  { -3.0837503404959957e+97, 100, 4.3000000000000007 },
  { 5.2136163919277588e+96, 100, 4.4000000000000004 },
  { 7.8073853756187101e+97, 100, 4.5000000000000000 },
  { 4.2440218943299170e+97, 100, 4.5999999999999996 },
  { -1.6545238014142650e+98, 100, 4.6999999999999993 },
  { -2.3193021196802549e+98, 100, 4.8000000000000007 },
  { 2.5601172475776894e+98, 100, 4.9000000000000004 },
  { 8.1931937130591466e+98, 100, 5.0000000000000000 },
  { -4.0259235416602546e+97, 100, 5.0999999999999996 },
  { -2.3096466887887237e+99, 100, 5.1999999999999993 },
  { -1.8337906983005823e+99, 100, 5.3000000000000007 },
  { 5.0935670879056567e+99, 100, 5.4000000000000004 },
  { 9.9309186425970402e+99, 100, 5.5000000000000000 },
  { -6.2933057672779134e+99, 100, 5.5999999999999996 },
  { -3.6379102593330386e+100, 100, 5.6999999999999993 },
  { -1.5140717605507249e+100, 100, 5.8000000000000007 },
  { 1.0141407690966954e+101, 100, 5.9000000000000004 },
  { 1.5170693933260738e+101, 100, 6.0000000000000000 },
  { -1.8039133351602961e+101, 100, 6.1000000000000014 },
  { -7.0672139150705246e+101, 100, 6.1999999999999993 },
  { -1.4587350659144549e+101, 100, 6.3000000000000007 },
  { 2.3341289432998748e+102, 100, 6.3999999999999986 },
  { 3.2072472002047670e+102, 100, 6.5000000000000000 },
  { -4.8039047085582927e+102, 100, 6.6000000000000014 },
  { -1.7849709684883083e+103, 100, 6.6999999999999993 },
  { -3.7817730019120996e+102, 100, 6.8000000000000007 },
  { 6.4900499886056430e+103, 100, 6.8999999999999986 },
  { 1.0002273523275075e+104, 100, 7.0000000000000000 },
  { -1.2947583568693485e+104, 100, 7.1000000000000014 },
  { -5.9348804074874565e+104, 100, 7.1999999999999993 },
  { -2.9237691057264679e+104, 100, 7.3000000000000007 },
  { 2.1375121759799924e+105, 100, 7.3999999999999986 },
  { 4.4772364475394960e+105, 100, 7.5000000000000000 },
  { -2.7486638813241244e+105, 100, 7.6000000000000014 },
  { -2.4850432648123868e+106, 100, 7.6999999999999993 },
  { -2.7252309851017070e+106, 100, 7.8000000000000007 },
  { 7.3445622927434568e+106, 100, 7.8999999999999986 },
  { 2.5761505535443451e+107, 100, 8.0000000000000000 },
  { 8.4932594446944218e+106, 100, 8.1000000000000014 },
  { -1.1534178973583771e+108, 100, 8.1999999999999993 },
  { -2.5611278359589723e+108, 100, 8.3000000000000007 },
  { 1.2481735250538004e+108, 100, 8.3999999999999986 },
  { 1.6203702280499544e+109, 100, 8.5000000000000000 },
  { 2.5386102824684747e+109, 100, 8.6000000000000014 },
  { -3.8930624477368766e+109, 100, 8.6999999999999993 },
  { -2.2669773300293168e+110, 100, 8.8000000000000007 },
  { -2.6071876743328939e+110, 100, 8.8999999999999986 },
  { 7.7294136683043515e+110, 100, 9.0000000000000000 },
  { 3.3341330166544429e+111, 100, 9.1000000000000014 },
  { 2.9686037583393142e+111, 100, 9.1999999999999993 },
  { -1.3977488190034162e+112, 100, 9.3000000000000007 },
  { -5.3374224541141370e+112, 100, 9.3999999999999986 },
  { -4.2265051766934789e+112, 100, 9.5000000000000000 },
  { 2.4904626130430740e+113, 100, 9.6000000000000014 },
  { 9.4857435427570856e+113, 100, 9.6999999999999993 },
  { 8.3761676654305186e+113, 100, 9.8000000000000007 },
  { -4.4232904120309469e+114, 100, 9.8999999999999986 },
  { -1.8738689223256612e+115, 100, 10.000000000000000 },
};

// Test function for n=100.
template<typename Tp>
  void
  test008()
  {
    bool test __attribute__((unused)) = true;
    const Tp eps = std::numeric_limits<Tp>::epsilon();
    Tp max_abs_diff = -Tp(1);
    Tp max_abs_frac = -Tp(1);
    unsigned int num_datum = sizeof(data008)
			   / sizeof(testcase_hermite<double>);
    for (unsigned int i = 0; i < num_datum; ++i)
      {
	const Tp f = std::tr1::hermite(Tp(data008[i].n), Tp(data008[i].x));
	const Tp f0 = data008[i].f0;
	const Tp diff = f - f0;
	if (std::abs(diff) > max_abs_diff)
	  max_abs_diff = std::abs(diff);
	if (std::abs(f0) > Tp(10) * eps
	 && std::abs(f) > Tp(10) * eps)
	  {
	    const Tp frac = diff / f0;
	    if (std::abs(frac) > max_abs_frac)
	      max_abs_frac = std::abs(frac);
	  }
      }
    VERIFY(max_abs_frac < Tp(2.5000000000000020e-13));
  }

int
main()
{
  test001<double>();
  test002<double>();
  test003<double>();
  test004<double>();
  test005<double>();
  test006<double>();
  test007<double>();
  test008<double>();
  return 0;
}
