/*
 * Copyright (C) 2006-2008 the VideoLAN team
 *
 * This file is part of VLMa.
 *
 * VLMa is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * VLMa is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with VLMa. If not, see <http://www.gnu.org/licenses/>.
 *
 */

package org.videolan.vlma.model;

import java.io.Serializable;
import java.net.InetAddress;
import java.util.Date;



/**
 * Basic implementation of Program which represents a programmation.
 *
 * @author Sylvain Cadilhac <sylv at videolan.org>
 */
public class Program implements Serializable {

    private static final long serialVersionUID = -2723021707121558896L;

    private boolean broadcastState;

    private InetAddress ip;

    private int liveInterval;

    private int liveLength;

    private int liveRepetitions;

    private Date liveStart;

    private InetAddress player;

    private int priority;

    private String sapGroup;

    private String sapName;

    private StreamingStrategy streamingStrategy;

    public Program() {
        setLiveInterval(-1);
        setLiveRepetitions(1);
        setLiveLength(-1);
        liveStart = new Date();
        setIp(null);
        setPriority(50);
        setPlayer(null);
    }

    /**
     * Gets broadcast state.
     *
     * @return The brocast state :
     *         True if the media is broadcast
     *         False if not
     */
    public boolean getBroadcastState() {
        return broadcastState;
    }

    /**
     * Gets the IP address used to stream the program.
     *
     * @return the IP address
     */
    public InetAddress getIp() {
        return ip;
    }

    /**
     * Gets the repetition interval of the programmation.
     *
     * @return the repetition interval
     */
    public int getLiveInterval() {
        return liveInterval;
    }

    /**
     * Gets the programmation length.
     *
     * @return the programmation length
     */
    public int getLiveLength() {
        return liveLength;
    }

    /**
     * Gets the number of repetitions of the programmation.
     *
     * @return the number of repetitions
     */
    public int getLiveRepetitions() {
        return liveRepetitions;
    }

    /**
     * Gets the programmation start time.
     *
     * @return the start time
     */
    public Date getLiveStart() {
        return liveStart;
    }

    /**
     * Gets the IP address of the server which streams on the program IP
     * address.
     *
     * @return the server IP address
     */
    public InetAddress getPlayer() {
        return player;
    }

    /**
     * Gets the programmation priority.
     *
     * @return the programmation priority
     */
    public int getPriority() {
        return priority;
    }

    /**
     * Gets the SAP group.
     *
     * @return the sap group
     */
    public String getSapGroup() {
        return sapGroup;
    }

    /**
     * Gets the SAP name which will be used during the streaming.
     *
     * @return the SAP name
     */
    public String getSapName() {
        return sapName;
    }

    /**
     * Gets the program streamingStrategy
     *
     * @return the streamingStrategy
     */
    public StreamingStrategy getStreamingStrategy() {
        return streamingStrategy;
    }

    @Override
    public boolean equals(Object other) {
        if (other == null || !(other instanceof Program))
            return false;
        Program p = (Program) other;
        return this.getSapGroup().equals(p.getSapGroup())
                && this.getSapName().equals(p.getSapName());
    }

    @Override
    public int hashCode() {
        int result = 0;
        result += sapGroup == null ? 0 : sapGroup.hashCode();
        result += sapGroup == null ? 0 : sapName.hashCode();
        return result;
    }

    /**
     * Returns true if diffusion has to take place.
     *
     * @return true if diffusion has to take place; false otherwise
     */
    public boolean isTimeToPlay() {
        long now = System.currentTimeMillis();
        long when = getLiveStart().getTime();
        if (getLiveInterval() > 0) {
            return (now >= when)
                    && ((now - when) % getLiveInterval() < getLiveLength());
        } else {
            return (now >= when);
        }
    }

    /**
     * Tells whether the media of the order is really broadcasted.
     *
     * @param broadcastState true if the media is broadcasted, false otherwise
     */
    public void setBroadcastState(boolean broadcastState) {
        this.broadcastState = broadcastState;
    }

    /**
     * Sets the IP address used to stream the program.
     *
     * @param ip
     *            the IP address to set
     */
    public void setIp(InetAddress ip) {
        this.ip = ip;
    }

    /**
     * Sets the repetition interval of the programmation.
     *
     * @param liveInterval
     *            the repetition interval
     */
    public void setLiveInterval(int liveInterval) {
        this.liveInterval = liveInterval;
    }

    /**
     * Sets the programmation length.
     *
     * @param liveLength
     *            the programmation length
     */
    public void setLiveLength(int liveLength) {
        this.liveLength = liveLength;
    }

    /**
     * Sets the number of repetitions of the programmation.
     *
     * @param liveRepetitions the number of repetitions
     *
     */
    public void setLiveRepetitions(int liveRepetitions) {
        this.liveRepetitions = liveRepetitions;
    }

    /**
     * Sets the programmation start time.
     *
     * @param liveStart
     *            the start time to set
     */
    public void setLiveStart(Date liveStart) {
        this.liveStart = liveStart;
    }

    /**
     * Sets the IP address of the server which streams on the program IP
     * address.
     *
     * @param player
     *            the server IP address
     */
    public void setPlayer(InetAddress player) {
        this.player = player;
    }

    /**
     * Sets the programmation priority.
     *
     * @param priority
     *            the priority to set
     */
    public void setPriority(int priority) {
        this.priority = priority;
    }

    /**
     * Sets the SAP group.
     *
     * @param sapGroup
     *            the SAP group to set
     */
    public void setSapGroup(String sapGroup) {
        this.sapGroup = sapGroup;
    }

    /**
     * Sets the SAP name which will be used during the streaming.
     *
     * @param sapName
     *            the SAP name to set
     */
    public void setSapName(String sapName) {
        this.sapName = sapName;
    }

    /**
     * Sets the streamingStrategy.
     *
     * @param streamingStrategy the streamingStrategy to set
     */
    public void setStreamingStrategy(StreamingStrategy streamingStrategy) {
        this.streamingStrategy = streamingStrategy;
    }
}
