/*******************************************************************************
* ps2ts.cpp: threaded PS to TS converter
*-------------------------------------------------------------------------------
* (c)1999-2001 VideoLAN
* $Id: ps2ts.cpp,v 1.2 2001/11/29 16:11:42 bozo Exp $
*
* Authors: Arnaud de Bossoreille de Ribou <bozo@via.ecp.fr>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*
*-------------------------------------------------------------------------------
*
*******************************************************************************/


//------------------------------------------------------------------------------
// Preamble
//------------------------------------------------------------------------------
#include "../../core/defs.h"

#include "../../core/core.h"

#include "../../mpeg/mpeg.h"
#include "../../mpeg/ts.h"
#include "../../mpeg/streamdescr.h"
#include "../../mpeg/ps2ts.h"

#include "../../server/buffer.h"
#include "../../server/program.h"
#include "../../server/broadcast.h"
#include "../../server/request.h"
#include "../../server/input.h"

#include "../../mpeg/reader.h"
#include "../../mpeg/converter.h"

#include "ps2ts.h"


#include "../../mpeg/ps2ts.cpp"


//------------------------------------------------------------------------------
// Library declaration
//------------------------------------------------------------------------------
#ifdef __PLUGIN__
GENERATE_LIB_ARGS(C_Ps2TsMpegConverterModule, handle);
#endif


//------------------------------------------------------------------------------
// Builtin declaration
//------------------------------------------------------------------------------
#ifdef __BUILTIN__
C_Module* NewBuiltin_ps2ts(handle hLog)
{
  return new C_Ps2TsMpegConverterModule(hLog);
}
#endif


/*******************************************************************************
* C_Ps2TsMpegConverter
********************************************************************************
*
*******************************************************************************/


//------------------------------------------------------------------------------
// Constructor
//------------------------------------------------------------------------------
C_Ps2TsMpegConverter::C_Ps2TsMpegConverter(C_Module* pModule,
                                        C_MpegConverterConfig& cConfig) :
        C_MpegConverter(pModule, cConfig),
        m_cConverter(cConfig.m_pReader,
                     cConfig.m_pTsProvider, 2,
                     cConfig.m_pBroadcast->GetOption("mpegversion").ToInt())
{
  m_bPreParse = (cConfig.m_pBroadcast->GetOption("preparse") == "1");

  LogDbg(m_hLog,
         C_String("Mpeg version : ") +
         cConfig.m_pBroadcast->GetOption("mpegversion").ToInt());
  LogDbg(m_hLog, C_String("Preparsing : ") + (m_bPreParse ? "yes" : "no"));
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
C_ProgramDescriptor* C_Ps2TsMpegConverter::GetPgrmDescriptor()
{
  return m_cConverter.GetPgrmDescriptor();
}


//------------------------------------------------------------------------------
// Initialization
//------------------------------------------------------------------------------
void C_Ps2TsMpegConverter::InitWork()
{
  C_MpegConverter::InitWork();

  m_cConverter.Synch();

  int iRc = 0;

  if(m_bPreParse)
  {
    for(unsigned int i = 0; i < m_pReader->Size() / 188 / 256; i++)
    {
      // preparsing to make the PAT and the PMT
      C_TsPacket* pPacket = m_cConverter.GetPacket();
      ASSERT(pPacket);
      m_pTsProvider->ReleasePacket(pPacket);
    }

    // Back to the beginning
    m_pReader->Seek(0, FILE_SEEK_BEGIN);
    m_cConverter.Synch();
  }

  for(unsigned int ui = 1; ui < m_pBuffer->Capacity(); ui++)
  {
    C_TsPacket* pPacket = m_cConverter.GetPacket();
    ASSERT(pPacket);
    m_pBuffer->Push(pPacket);
    if(m_cConverter.GetStatus())
    {
      iRc = MPEG_STREAMERROR;
      break;
    }
  }

  if(iRc == MPEG_ENDOFSTREAM)
    iRc = NO_ERR;
  else if(iRc)
    throw E_Exception(GEN_ERR, "PS to TS converter failed to start");
}


//------------------------------------------------------------------------------
// Main job
//------------------------------------------------------------------------------
void C_Ps2TsMpegConverter::DoWork()
{
  C_String strPgrmName = m_pBroadcast->GetProgram()->GetName();

  LogDbg(m_hLog, "Starting to read program \"" + strPgrmName + "\"");

  int iRc = 0;
  bool bDiscontinuity = false;

  // Stream the file
  while(!m_bStop && !iRc)
  {
    // Suspend/Resume handling
    if(m_iShortPauseStatus == STATUS_PAUSE_REQUEST)
    {
      m_cResumeCond.Wait();
      m_iShortPauseStatus = STATUS_RUNNING;
    }

    if(m_iLongPauseStatus == STATUS_PAUSE_REQUEST)
    {
      m_cResumeCond.Wait();
      m_iLongPauseStatus = STATUS_RUNNING;
      bDiscontinuity = true;
    }

    // Get a packet to fill
    C_TsPacket* pPacket = m_cConverter.GetPacket();

    switch(m_cConverter.GetStatus())
    {
    case 0:
      if(!pPacket)
        iRc = MPEG_STREAMERROR;
      break;
    case -99:
      iRc = MPEG_ENDOFSTREAM;
      break;
    default:
      iRc = MPEG_STREAMERROR;
      break;
    }

    // Check stream discontinuity
    if(    !iRc
        && (m_pReader->HasDiscontinuity() || bDiscontinuity)
        && pPacket->HasPCR())
    {
      ASSERT(pPacket);
      ASSERT(pPacket->SetDiscontinuityFlag());
      pPacket->SetDiscontinuityFlag();
      m_pReader->ResetDiscontinuity();
      bDiscontinuity = false;
    }

    if(!iRc)
    {
      // Stores the packet in the buffer
      m_pBuffer->Push(pPacket);
    }
    else if(iRc == MPEG_ENDOFSTREAM)
    {
      C_String strPgrmName = m_pBroadcast->GetProgram()->GetName();
      LogDbg(m_hLog, "End of program \"" + strPgrmName + "\" reached");
      if(pPacket)
        m_pTsProvider->ReleasePacket(pPacket);
    }
    else
    {
      C_String strPgrmName = m_pBroadcast->GetProgram()->GetName();
      Log(m_hLog, LOG_ERROR, "Read error for program \"" + strPgrmName + "\"");
      if(pPacket)
        m_pTsProvider->ReleasePacket(pPacket);
    }
  }

  if(!m_bStop)
  {
    LogDbg(m_hLog, "Stopping converter by callback for program " + strPgrmName);
    m_bStop = true;
    C_Event cEvent(m_pBroadcast->GetInput()->GetName());
    cEvent.SetCode(EOF_EVENT);
    cEvent.SetBroadcast(m_pBroadcast);
    m_pEventHandler->HandleEvent(cEvent);
  }

  LogDbg(m_hLog, "Converter stopped for program " + strPgrmName);
}


