/*******************************************************************************
* telnet.cpp:
*-------------------------------------------------------------------------------
* (c)1999-2001 VideoLAN
* $Id: telnet.cpp,v 1.7 2002/08/14 11:13:27 tooney Exp $
*
* Authors: Benoit Steiner <benny@via.ecp.fr>
*          Arnaud de Bossoreille de Ribou <bozo@via.ecp.fr>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*
*-------------------------------------------------------------------------------
* Informations on the Telnet protocol can be found in RFC 854 to 861 and also
* RFC 1184 for linemodes
*
*******************************************************************************/



//------------------------------------------------------------------------------
// Preamble
//------------------------------------------------------------------------------
#include "../core/defs.h"

#include <ctype.h>

#include "config.h"
#include "../core/core.h"
#include "../core/network.h"
#include "request.h"
#include "admin.h"
#include "telnet.h"


#include "../core/network.cpp"


//------------------------------------------------------------------------------
// Telnet interface
//------------------------------------------------------------------------------
#define WELCOME         "Videolan Server Administration System"
#define LOGIN           "Login: "
#define PASSWD          "Password: "
#define PROMPT          "vls> "


//------------------------------------------------------------------------------
// Telnet protocol constants
//------------------------------------------------------------------------------
#define IAC     255             // Interpret as command
#define DONT    254             // You are not to use option
#define DO      253             // Please use option
#define WONT    252             // I won't use option
#define WILL    251             // I will use option
#define EL      248             // Erase the current line

#define TELOPT_ECHO     1       // Echo
#define TELOPT_SGA      3       // Suppress go ahead
#define TELOPT_OUTMRK   27      // Output marking
#define TELOPT_LINEMODE 34      // Linemode option

#define KEY_UP          65      // 
#define KEY_DOWN        66      // 
#define KEY_LEFT        67      // 
#define KEY_RIGHT       68      //


//------------------------------------------------------------------------------
// Telnet module definitions
//------------------------------------------------------------------------------
#define CONFIG_MODE             1
#define CONTROL_MODE            2
#define INTERACTIVE_MODE        3

#define INIT_PHASE              1
#define LOGIN_PHASE             2
#define PASSWD_PHASE            3
#define COMMAND_PHASE           4

#define HISTORY_SIZE            20



/*******************************************************************************
* E_Telnet exception
********************************************************************************
*
*******************************************************************************/
E_Telnet::E_Telnet(const C_String& strMsg) : E_Exception(GEN_ERR, strMsg)
{
}

E_Telnet::E_Telnet(const C_String& strMsg, const E_Exception& e) :
             E_Exception(GEN_ERR, strMsg, e)
{
}


/*******************************************************************************
* C_TelnetOptions class
********************************************************************************
* Manage the configuration of the 2 NVT through telnet negociation
* Very basic at that time, is to be completly rewritten
*******************************************************************************/

//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
C_Buffer<byte> C_TelnetOptions::Request(byte bAction, byte bOption)
{
  C_Buffer<byte> cAnswer(3);
  cAnswer.SetSize(3);
  cAnswer[0] = IAC;
  cAnswer[1] = bAction;
  cAnswer[2] = bOption;
  
  //m_cPendingRequests.PushEnd(new C_Buffer<byte>(cAnswer));

  return cAnswer;
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
C_Buffer<byte> C_TelnetOptions::Answer(const C_Buffer<byte>& cRequest)
{
  ASSERT(cRequest.GetSize() == 3);
  
  C_Buffer<byte> cAnswer(3);
  cAnswer.SetSize(0);
  
  //  cAnswer.SetSize(3);
  //  cAnswer[0] = IAC;
  //  cAnswer[2] = cRequest[3];

  return cAnswer;
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
int C_TelnetOptions::GetOptionStatus()
{
  return YES;
}





/*******************************************************************************
* C_TelnetCmdLine class
********************************************************************************
* Manage a telnet command line
*******************************************************************************/

//------------------------------------------------------------------------------
// Constructor
//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
C_TelnetCmdLine::C_TelnetCmdLine() : m_cCurrent(m_cHistory.CreateIterator())
{
  m_iPosInLine = 0;
}


//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
C_String C_TelnetCmdLine::Append(byte bByte)
{
  C_String cResult;

  switch(bByte)
  {
    case 0x7F:
    {
      if(m_strLine.Length() > 0)
      {
        m_strLine = m_strLine.SubString(0, m_strLine.Length()-1);
        cResult += "\b\e[P";
      }
      break;
    }
    default:
    {
      if(bByte != '\n' && bByte != '\0')
      {
        m_strLine += (char)bByte;
        cResult += (char)bByte;
      }
    }
  }
  
  return cResult;
}


//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
C_Buffer<byte> C_TelnetCmdLine::Edit(const C_Buffer<byte>& cCmd)
{
  ASSERT(cCmd.GetSize() == 3);
  ASSERT(cCmd[0] = TELOPT_OUTMRK);

  C_String strResult;

  switch(cCmd[2])
  {
    case KEY_UP:
    {
      if(m_cCurrent.HasNext())
      {
         strResult = *m_cCurrent.GetNext();
      }
      break;
    }
    case KEY_DOWN:
    {
      if(m_cCurrent.HasPrevious())
      {
        strResult = *m_cCurrent.GetPrevious();
      }
      break;
    }
  }

  if(strResult != "")
  {
    StartNewLine();
    m_strLine = strResult;
  }

  return /*EL + strResult*/C_Buffer<byte>(0);
}


//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
C_String C_TelnetCmdLine::GetCmdLine()
{
  // Strip line
  C_String strResult = m_strLine.Strip(" \t");
  return strResult;
}


//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
void C_TelnetCmdLine::StartNewLine()
{
  // Store current line if not empty
  C_String strLine = GetCmdLine();
  if(strLine.Length() > 0)
  {
    m_cHistory.PushStart(new C_String(strLine));
    // To do: removal of old commands
  }
  
  // Reset command history
  m_cCurrent.Reset();

  // Empty current line
  m_strLine = "";
}


//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
void C_TelnetCmdLine::DropLine()
{
  m_strLine = "";
}




/*******************************************************************************
* C_TelnetSession class
********************************************************************************
* 
* 
*******************************************************************************/

//------------------------------------------------------------------------------
// Constructor
//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
C_TelnetSession::C_TelnetSession(C_Socket* pConnection, void* pAdmin) :
                      C_AdminSession((C_Admin*)pAdmin),
                      m_strPeerName("[Unknown]"), m_cStream(pConnection),
                      m_cBuffer(3)
{
  ASSERT(pConnection);

  m_pConnection = pConnection;
  
  // Starting mode
  m_iPhase = INIT_PHASE;
  m_iMode = INTERACTIVE_MODE;
}


//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
void C_TelnetSession::OnInit()
{
  // Actual initialisation is done internally in small pieces cause it could
  // block the entire server if done in one single step. For now just send
  // the welcome message
  m_cStream << "\r\n" << WELCOME << "\r\n\r\n";

  // Transmission parameters: to be able to deal with all clients, editing is
  // done on the server
  m_cStream << m_cOptions.Request(WILL, TELOPT_ECHO);
  m_cStream << m_cOptions.Request(WILL, TELOPT_SGA);
  m_cStream << m_cOptions.Request(DONT, TELOPT_LINEMODE);

  m_cStream << LOGIN;
  m_iPhase = LOGIN_PHASE;
}


//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
void C_TelnetSession::OnClose()
{
  try
  {
    // Send a goodby message if possible
    if(m_cStream.GetState() == STREAM_READY)
      m_cStream << "\r\nClosing connection" << "\r\n";

    // Close the connection
    m_pConnection->Close();
  }
  catch(E_Exception e)
  {
    throw E_Telnet("Unable to close connection with "+m_strPeerName, e);
  }
}


//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
void C_TelnetSession::ProcessData()
{
  try
  {
    // First get the data from the network
    C_Buffer<byte> cData(256);
    m_cStream >> cData;
    // There must be some data because the select told us else the client
    // closed his session.
    if(cData.GetSize() <= 0)
      throw E_Telnet("Connection reset by peer");

    unsigned int iPos = 0;
    unsigned int iSize = cData.GetSize();
  
    while(iPos < iSize)
    {
      switch(cData[iPos])
      {
        case IAC:
        {
          // Telnet negociation or special char
//          printf("negociation: %x %x\n", cData[iPos+1], cData[iPos+2]);
          m_cBuffer[0] = IAC;
          m_cBuffer.SetSize(1);
          m_iMode = CONFIG_MODE;
          break;
        }

        case TELOPT_OUTMRK:
        {
          // Telnet special chars
//          printf("output marking: %x %x\n", cData[iPos+1], cData[iPos+2]);
          m_cBuffer[0] = TELOPT_OUTMRK;
          m_cBuffer.SetSize(1); 
          m_iMode = CONTROL_MODE;
          break;
        }

        case '\r':
        {
          // End of line sent: handle the corresponding data
          HandleLine();
        
          // Send prompt for next line
          m_cStream << "\r\n" << m_strPrompt;
          break;
        }
      
        default:
        {
          HandleByte(cData[iPos]);
          break;
        }
      }

      // Go to next char in input
      iPos++;
    }
  }
  catch(E_Stream<C_Socket> e)
  {
    throw E_Telnet("Connection error", e);
  }
}


//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
void C_TelnetSession::HandleLine()
{
  switch(m_iPhase)
  {
    case LOGIN_PHASE:
    {
      // Execute first login phase
      ExecLogin();
      // Drop line since we don't have to remember login
      m_cCmdLine.DropLine();
      break;
    }
    case PASSWD_PHASE:
    {
      // Check login
      ExecLogin();
      // Drop line since we don't have to remember passwd
      m_cCmdLine.DropLine();
      break;
    }
    default:
    {
      ASSERT(m_iPhase == COMMAND_PHASE);
      // Execute command
      ExecCommand();
      // Start a new line
      m_cCmdLine.StartNewLine();
      break;
    }
  }
}


//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
void C_TelnetSession::HandleByte(byte bByte)
{
  //printf("char: '%c' (%x)\n", bByte, bByte);
  
  switch(m_iMode)
  {
    case CONFIG_MODE:
    {
      ASSERT(m_cBuffer[0] == IAC);
      unsigned int iSize = m_cBuffer.GetSize();
      m_cBuffer[iSize] = bByte;
      iSize++;
      m_cBuffer.SetSize(iSize);
      if(iSize == 3)
      {
        m_cStream << m_cOptions.Answer(m_cBuffer);
        m_cBuffer.SetSize(0);
        m_iMode = INTERACTIVE_MODE;
      }
      break;
    }
    case CONTROL_MODE:
    {
      ASSERT(m_cBuffer[0] == TELOPT_OUTMRK);
      unsigned int iSize = m_cBuffer.GetSize();
      m_cBuffer[iSize] = bByte;
      iSize++;
      m_cBuffer.SetSize(iSize);
      if(iSize == 3)
      {
        m_cStream << m_cCmdLine.Edit(m_cBuffer);
        m_cBuffer.SetSize(0);
        m_iMode = INTERACTIVE_MODE;
      }
      break;
    }
    default:
    {
      ASSERT(m_iMode == INTERACTIVE_MODE);

      // Drop all characters before init is completed
      int iStatus = m_cOptions.GetOptionStatus();
      if(iStatus == YES)
      {
        // Remote echo      
        C_String cRemoteCmdLine = m_cCmdLine.Append(bByte);
        if(m_iPhase != PASSWD_PHASE)
          m_cStream << cRemoteCmdLine;
      }
      else if(iStatus == NO)
        throw E_Telnet("Client not supported");
      
      //printf("char: %c\n", bByte);
      break;
    }
  }
}


//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
void C_TelnetSession::ExecLogin()
{
  ASSERT(m_iPhase == LOGIN_PHASE || m_iPhase == PASSWD_PHASE);

  if(m_iPhase == LOGIN_PHASE)
  {
    // This is the login which has just been entered
    m_strLogin = m_cCmdLine.GetCmdLine();
    m_strPrompt = PASSWD;
    m_iPhase = PASSWD_PHASE;
  }
  else
  {
    // Login has been entered before, this is the passwd
    C_String strPasswd = m_cCmdLine.GetCmdLine();
    
    // Authentication (to do)
    int iRc = Authenticate(m_strLogin, strPasswd);

    if(!iRc)
    {
      m_strPrompt = m_strLogin + "@" + PROMPT;
      m_iPhase = COMMAND_PHASE;
    }
    else
    {
      m_strPrompt = LOGIN;
      m_iPhase = LOGIN_PHASE;
    }

    m_cStream << "\r\n";
  }
}


//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
void C_TelnetSession::ExecCommand()
{
  ASSERT(m_iPhase == COMMAND_PHASE);

  C_String strCmd = m_cCmdLine.GetCmdLine();

  C_Request cRequest("");
  C_Answer cAdminAnswer = m_pAdmin->ParseCmdLine(this, strCmd, cRequest);
  switch(cAdminAnswer.GetStatus())
  {
  case ADMIN_WELLFORMED_COMMAND:
    if(cRequest.GetCmd() == "logout")
    {
      throw E_Telnet("logout requested");
    }
    else
    {
      C_Answer cAnswer = m_pAdmin->HandleRequest(cRequest);
      SendAnswer(cAnswer);
    }
    break;
  case ADMIN_COMMAND_NOT_VALID:
  case ADMIN_UNKNOWN_COMMAND:
  case ADMIN_COMMAND_DENIED:
    SendAnswer(cAdminAnswer);
    break;
  case ADMIN_EMPTY_COMMAND:
    // Nothing to do
    break;
  default:
    ASSERT(false);
    break;
  }
}


//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
// Send a C_Answer in human readable format
//------------------------------------------------------------------------------
void C_TelnetSession::SendAnswer(const C_Answer& cAnswer,
                                 const C_String& strPrefix)
{
  if(cAnswer.GetProvider() != "")
  {
    m_cStream << "\r\n";
    m_cStream << strPrefix + "Provider: " + cAnswer.GetProvider() + "\r\n";

    int iStatus = cAnswer.GetStatus();
    if(iStatus >= 0)
      m_cStream << strPrefix + "Status: " + iStatus + "\r\n";
    else
      m_cStream << strPrefix + "Error: " + iStatus + "\r\n";

    C_List<C_String> cMessageList = cAnswer.GetMessages();
    unsigned int iSize = cMessageList.Size();
    for(unsigned int i = 0; i < iSize; i++)
    {
      m_cStream << strPrefix + "Info: " + cMessageList[i] + "\r\n";
    }

    C_List<C_Answer> cAnswerList = cAnswer.GetSubAnswers();
    iSize = cAnswerList.Size();
    for (unsigned int j = 0; j < iSize; j++)
    {
      SendAnswer(cAnswerList[j], strPrefix+"  ");
    }
  }
}





/*******************************************************************************
* C_Telnet class
********************************************************************************
* 
*******************************************************************************/

//------------------------------------------------------------------------------
// Constructor
//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
C_Telnet::C_Telnet(handle hLog, C_Admin *pAdmin) :
              C_ConnectionsHandler<C_TelnetSession>(hLog, pAdmin)
{
  ASSERT(hLog);
  ASSERT(pAdmin);

  m_hLog = hLog;
  m_pAdmin = pAdmin;
}


//------------------------------------------------------------------------------
// Initialization
//------------------------------------------------------------------------------
int C_Telnet::Init()
{
  C_Application* pApp = C_Application::GetApp();
  ASSERT(pApp);

  int iRc = NO_ERR;

  // Get config
  C_String strDomain = pApp->GetSetting("Telnet.Domain", "Inet4").ToLower();

  int iDomain;
  C_String strDefaultHost;

  if(strDomain == "inet4")
  {
    iDomain = AF_INET;
    strDefaultHost = "0.0.0.0";
  }
#ifdef HAVE_IPV6
  else if(strDomain == "inet6")
  {
    iDomain = AF_INET6;
    strDefaultHost = "0::0";
  }
#endif
  else
  {
    Log(m_hLog, LOG_ERROR, "Unknown domain:\n" + strDomain);
    iRc = GEN_ERR;
  }

  if(!iRc)
  {
    C_String strAddr = pApp->GetSetting("Telnet.LocalAddress", strDefaultHost);
    C_String strPort = pApp->GetSetting("Telnet.LocalPort", "9999");

    // Init the socket
    try
    {
      C_ConnectionsHandler<C_TelnetSession>::Init(iDomain, strAddr, strPort);
    }
    catch(E_Exception e)
    {
      Log(m_hLog, LOG_ERROR,
          "Telnet server initialisation failed:\n" + e.Dump());
      iRc = GEN_ERR;
    }
  }
  
  if(!iRc)
    Log(m_hLog, LOG_NOTE, "Telnet server initialised");

  return iRc;
}


//------------------------------------------------------------------------------
// Execution
//------------------------------------------------------------------------------
int C_Telnet::Run()
{
  int iRc = NO_ERR;

  try
  {
    C_ConnectionsHandler<C_TelnetSession>::Run();
  }
  catch(E_Exception e)
  {
    iRc = GEN_ERR;
    Log(m_hLog, LOG_ERROR, "Telnet server launch failed:\n" + e.Dump());
  }

  return iRc;
}


//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
int C_Telnet::Stop()
{
  int iRc = NO_ERR;

  try
  {
    C_ConnectionsHandler<C_TelnetSession>::Stop();
  }
  catch(E_Exception e)
  {
    Log(m_hLog, LOG_ERROR, "Could not stop telnet server:\n" + e.Dump());
    iRc = GEN_ERR;
  }

  if(!iRc)
    Log(m_hLog, LOG_NOTE, "Telnet server stopped");

  return iRc;
}


//------------------------------------------------------------------------------
// Destruction
//------------------------------------------------------------------------------
int C_Telnet::Destroy()
{  
  int iRc = NO_ERR;

  try
  {
    C_ConnectionsHandler<C_TelnetSession>::Destroy();
  }
  catch(E_Exception e)
  {
    Log(m_hLog, LOG_ERROR, "Error during telnet server destruction:\n" + e.Dump());
    iRc = GEN_ERR;
  }

  if(!iRc)
    Log(m_hLog, LOG_NOTE, "Telnet server destroyed");

  return iRc;
}

